static const char _cvsid[] =
"$Id: FFTResample0.cc,v 1.3 2004/04/26 16:28:00 maciek Exp $";
static const char _copyright[] = 
"Copyright 2004, University of Wisconsin Space Science & Engr. Center."; 

/**
 \file FFTResample0.cc
 
 \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>

 This is a first-order implementation of a linear spectral resampling algorithm
 appropriate for use with the GIFTS and other interferometers. It is specified as
 an operator to be embedded into a pluggable processing stage capable of operating
 in more than one testing or deployment frameworks. 
 
 Its reference implementation is a set of matlab routines devised by David Tobin,
 and it is based on the current state of processing art, on instruments like S-HIS and NAST-I.
 
Dependencies:
 fftw2 - Open source Fast Fourier Transform library.
 
 */



/* C includes */
#include <cassert>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* STL includes */
#include <string>
#include <vector>

/* FFTW includes */
#include <fftw.h>
#include <rfftw.h>

/* other includes */
#include <FBF.h>
//#include <blitz/array.h>

#ifdef PROFILING
// clok.cc and clok.h from cvs TOOLS/dev/librkg/time/ 
#include "clok.h"
#endif

/* interface specification */
#include "FFTResampler.h"

using namespace std;
//using namespace blitz;

#ifdef SMW
static const size_t SPECTRUM_SIZE = 2049;
static const size_t IFG_SIZE = 4096;
static const size_t UPSAMPLED_IFG_SIZE = 65536;
static const size_t UPSAMPLED_SPECTRUM_SIZE = 32769;
static const double NORMALIZATION = 1.0 / 2048.0;
#else
static const size_t SPECTRUM_SIZE = 1025;
static const size_t IFG_SIZE = 2048;
static const size_t UPSAMPLED_IFG_SIZE = 65536;
static const size_t UPSAMPLED_SPECTRUM_SIZE = 32769;
static const double NORMALIZATION = 1.0 / 2048.0;
#endif
/************************************
* Implementation of the operator interface.
*/

/** \struct FFTResamplerInternals
* \brief Private data structures and utility functions used by the operator public interface.
*/
struct FFTResamplerInternals
{
    vector<double> spec_reflected;
    vector<double> short_ifg;

    vector<double> ifg_zerofilled;
    vector<double> zeros;
    vector<double> spec_upsampled_reflected;
    vector<double> spec_upsampled;

    rfftw_plan planforth;
    rfftw_plan planback;

    ReferenceDatabaseForFFTResampler &refdb;
    AuditingServiceForFFTResampler &audit;
    MonitoringServiceForFFTResampler &monitor;

    FFTResamplerInternals( ReferenceDatabaseForFFTResampler &_refdb,
                           AuditingServiceForFFTResampler &_audit,
                           MonitoringServiceForFFTResampler &_monitor): 
                     refdb(_refdb), audit(_audit), monitor(_monitor),
                     spec_reflected( IFG_SIZE, 0.0 ),
                     short_ifg( IFG_SIZE, 0.0 ),
                     ifg_zerofilled( UPSAMPLED_IFG_SIZE, 0.0 ),
                     zeros( UPSAMPLED_IFG_SIZE, 0.0 ),
                     spec_upsampled_reflected( UPSAMPLED_IFG_SIZE, 0.0 ),
                     spec_upsampled( UPSAMPLED_SPECTRUM_SIZE, 0.0 )   {

        planforth = rfftw_create_plan( IFG_SIZE,
                                       FFTW_REAL_TO_COMPLEX,
                                       FFTW_ESTIMATE );

        planback = rfftw_create_plan( UPSAMPLED_IFG_SIZE,
                                      FFTW_COMPLEX_TO_REAL,
                                      FFTW_ESTIMATE );
    }

    ~FFTResamplerInternals();

    inline double getVlaserRatio( const DetectorIndex_t key )
    {
        const ResamplingSettings_t *pRefData = refdb[key];
        return pRefData->sourceLaserWavenumber / pRefData->targetLaserWavenumber;
    }
};



int upsample( vector<double> &spec_upsampled, const double spec_in[],
              const rfftw_plan planforth, const rfftw_plan planback,
              FFTResamplerInternals *scratch )
{
    // reflect the spectrum
    copy( &spec_in[0], &spec_in[SPECTRUM_SIZE - 1], scratch->spec_reflected.begin() );
    copy( &spec_in[1], &spec_in[SPECTRUM_SIZE - 1], scratch->spec_reflected.rbegin() );

    // transform to halfcomplex interferogram
    rfftw_one( planforth, (fftw_real *)&scratch->spec_reflected[0],
                          (fftw_real *)&scratch->short_ifg[0] );

    // copy ends to larger halfcomplex interferogram vector.
    copy( scratch->zeros.begin(), scratch->zeros.end(), scratch->ifg_zerofilled.begin() );
    copy( scratch->short_ifg.begin(), scratch->short_ifg.begin() + SPECTRUM_SIZE,
          scratch->ifg_zerofilled.begin() );
    copy( scratch->short_ifg.rbegin(), scratch->short_ifg.rbegin() + SPECTRUM_SIZE - 2,
          scratch->ifg_zerofilled.rbegin() );
    scratch->ifg_zerofilled[SPECTRUM_SIZE-1] *= 0.5; // hack to overcome a possible rfftw bug.

    // re-transform to upsampled spectrum
    rfftw_one( planback, (fftw_real *)&scratch->ifg_zerofilled[0],
                         (fftw_real *)&scratch->spec_upsampled_reflected[0] );

    transform( scratch->spec_upsampled_reflected.begin(),
               scratch->spec_upsampled_reflected.begin() + UPSAMPLED_SPECTRUM_SIZE,
               spec_upsampled.begin(),
               bind2nd(multiplies<double>(), NORMALIZATION) );

    return 0;
}



int linear_downsample( double spec_out[], const vector<double> &spec_upsampled,
                       const double vlaser_ratio )
{
    double A = 32.0 * ( SPECTRUM_SIZE - 1 ) * ( 1 / vlaser_ratio - 1 );
    double B = 32.0 / vlaser_ratio;
    double r, rf; // fractional resample position
    int ri; // integral resample position
    for (unsigned int n = 0; n < SPECTRUM_SIZE; ++n )
    {
        r = A + B * n;
        ri = (int)floor(r);
        rf = r - ri;

        if (r < 0 || r >= UPSAMPLED_SPECTRUM_SIZE)
            spec_out[n] = 0;
        else
            spec_out[n] = spec_upsampled[ri]
                        + (spec_upsampled[ri+1] - spec_upsampled[ri]) * rf;
    }
    return 0;
}




FFTResampler::FFTResampler( ReferenceDatabaseForFFTResampler &_refdb, 
                              AuditingServiceForFFTResampler &_audit, 
                              MonitoringServiceForFFTResampler &_monitor ):
    my( *( new FFTResamplerInternals( _refdb, _audit, _monitor ) ) )
{
}


FFTResampler::~FFTResampler( )
{
    //delete &my;
}


const std::string &FFTResampler::errorString()
{
    abort( ); // FIXME: UNIMPLEMENTED
    static const std::string NYI( "Not Yet Implemented" );
    return NYI;
}

void FFTResampler::referenceDatabaseHasChanged()
{
}

const unsigned FFTResampler::CACHE_SIZE_UNLIMITED=unsigned(-1);

void FFTResampler::setMaxCacheSize( unsigned )
{
    abort(); // FIXME: UNIMPLEMENTED
}

/** Read max cache size in bytes.
*/
unsigned FFTResampler::maxCacheSize( ) const
{
    return CACHE_SIZE_UNLIMITED; // FIXME
}


int FFTResampler::operator()(  CalibratedRadianceSpectrum output, 
                                const CalibratedRadianceSpectrum input, 
                                DetectorIndex_t pixel )
{
    #ifdef PROFILING
        stopwatch.start();
    #endif
    int rc = upsample( my.spec_upsampled, input, my.planforth, my.planback, &my );
    const double vlaser_ratio = my.getVlaserRatio( pixel );
    rc += linear_downsample( output, my.spec_upsampled, vlaser_ratio ); 
    #ifdef PROFILING
        stopwatch.stop();
        stopwatch.result( N, wall, user, sys );
        my.monitor.timeCostOfResampling( wall, user, sys );
    #endif
    if (rc!=0) goto aborted;
    my.monitor.resamplingCompleted();
    return 0;
    
aborted:
    my.monitor.resamplingAborted();
    return rc;
}




#if TEST==1

int test1( int argc, char **argv )
{
    ResamplingMatrix F;
    /* test pattern matching stored results using arbitrary ratio src/dst = 1.1 */
    genFMatrix( F, 17, 1.1, 1.0 ); 
    cout << F;
}


int main( int argc, char **argv )
{
    return test1( argc, argv );
}
#endif

