/** 
*  \file FFTResampler.h
*  \brief Interface specification for Spectral Resampling Algorithm.
*    This is an implementation of a doulbe FFT upsampling approach,
*    as used by current FTS instruments such as S-HIS and NAST-I
*
*    This algorithm implementation is intended to be used in testing
*    error and perfomance tradoffs between it and an equivalent sin matrix resampler.
*
*    This operator requires a set of abstracted services which provide 
*    monitoring, auditing and reference database functionality, which
*    is deliberately out of scope of this interface spec. In practice,
*    this operator will be embedded into a larger processing stage which
*    implements these interfaces. 
*
*    based on SincResampler.h, by Ray Garcia <rayg@ssec.wisc.edu>
*
*  \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>
*
*  \version $Id: FFTResampler.h,v 1.1 2004/04/22 22:40:28 maciek Exp $
*
*  \par Revision Log:
*  \verbatim
*
*  $Log: FFTResampler.h,v $
*  Revision 1.1  2004/04/22 22:40:28  maciek
*  first take at ZFLI algorithm resampling ensemble, which conforms to
*  the SincResampler interface.
*  This version seems to work at first glance, but has not yet been tested
*  to any great extent.
*
*
*  \endverbatim
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
*  \endverbatim
*/

#ifndef H_FFTResampler
#define H_FFTResampler

#ifndef NO_DEPENDENCY_INCLUDES
/* Dependent headers for the interface.
 */

/* system includes */
#include <string>

/* Third party includes */
//#include <blitz/array.h>

/* Internal includes */
// #include "common.h"

#endif

/*
 * The following typedefs get moved to a common.h declaration file linking the underlying 
 * array/matrix representations to named types appropriate to the domain. 
 */

/* data type for elements of input and output spectra */
#ifdef USE_SINGLE_PRECISION
    typedef float data_t; 
#else
    typedef double data_t;
#endif

//typedef blitz::Array<data_t, 1> CalibratedRadianceSpectrum;
typedef double *CalibratedRadianceSpectrum;

/** Detector Index. 
 * Effectively a key into the reference database. 
 * Expect that this will be used for caching of resampling matrices.
 * Units: none.
 */
typedef unsigned long DetectorIndex_t;


/** 
 * \struct Resampling Settings structure
 * Reference Database structure for a detector used by FFTResampler.
 */
struct ResamplingSettings_t
/* is native byte order, IEEE floating point and integer */
/* Put this in a header file. And an interface control document. */
{ 
    /** source laser wavenumber -- varies per off-axis angle 
     * Units: cm-1
     */
    double sourceLaserWavenumber;
    /** target laser wavenumber -- generally constant 
     * Units: cm-1
     */
    double targetLaserWavenumber;
    /** number of points in the input spectrum -- e.g. 1025 for a 2048 interferogram 
     * Units: 
     */
    unsigned long inputPointCount; 
    /** quality reduction factor reduces the effective size of the resampling matrix at cost of error. 
     * Units:
     */
    unsigned long qualityReductionFactor;  // not used - kept for compatibility with SincResampler resampling db.
    /** decimation factor 
     * Units: 
     */
    unsigned long decimationFactor;  // not used.
};



/** FFT Resampler abstract interface to Auditing Service
 */
class AuditingServiceForFFTResampler
{
public:

    
};


/** \class MonitoringServiceForFFTResampler
 * \brief FFT Resampler abstract interface to Monitoring Service
 */
class MonitoringServiceForFFTResampler
{
public:
    /** completion of job
    */
    virtual void resamplingCompleted() = 0;
    
    /** failure of job
    */
    virtual void resamplingAborted() = 0;
    
    /** Here, a cache would be composed of precomputed linear interpolation regime vectors
    *   This is nowhere as expensive as the Sinc Matrix resampling cache.
    */
    virtual void addingEntryToResamplingCache( DetectorIndex_t forThisDetector, size_t resultantCacheSize ) = 0;
    
    /** Indicate that entry is being removed from the cache. 
    */
    virtual void removingEntryFromResamplingCache( DetectorIndex_t forThisDetector, size_t resultantCacheSize ) = 0;
    /** Indicate that cache flush has occurred. 
    */
    virtual void resamplingCacheFlushed( ) = 0;
    /** Indicate that a reference database fetch returned no information
    */
    virtual void referenceResamplingSettingsNotFound( DetectorIndex_t forThisDetector ) = 0;
    
    /** if compiled with PROFILING flag, these will be reported after the completion of an operation
     */
    virtual void timeCostOfResampling( double wall, double user, double system ) = 0;
};


/** \class ReferenceDatabaseForFFTResampler
 * \brief FFT Resampler abstract interface to Reference Database. 
 * Client interface to reference database used by FFT Resampler to fetch ResamplingSettings_t structs.
 */
class ReferenceDatabaseForFFTResampler 
{
public:

    /** Return reference database structure for a given detector, or NULL if none is available.
    */
    virtual const ResamplingSettings_t *operator[]( DetectorIndex_t ) = 0;
};


/** FFTResampler operator.
 * Instances of this object use the provided services to maintain an internal cache 
 * of transformations in order to resample the calibrated radiance spectra provided.
 */
class FFTResampler
{
private:
    struct FFTResamplerInternals &my;
    
public:

    /** Constructor
    */
    FFTResampler( ReferenceDatabaseForFFTResampler &, 
                   AuditingServiceForFFTResampler &, 
                   MonitoringServiceForFFTResampler & );
    /** Destructor
    */
    ~FFTResampler();
    
    /** Return a string describing an error code.
     */
    static const std::string &errorString(); 
    
    /** Invalidate cache contents. 
     * This informs the operator that the reference database has changed.
     */
    void referenceDatabaseHasChanged(); 
    
    /** Hint preferred max cache size in bytes. 
     *  May be rounded down to nearest convenient page size prior to use.
     */
    void setMaxCacheSize( unsigned );
    static const unsigned CACHE_SIZE_UNLIMITED; 
    
    /** Read max cache size in bytes.
    */
    unsigned maxCacheSize( ) const;
    
    /** Execute spectral resampling.
     * 
     *  \param output Output spectrum must be allocated identical in size to input but need not be initialized. 
     *  \param input Input spectrum corresponds to the sourceLaserWavenumber sampling stated in the resultant settings.
     *  \param pixel Detector Index must be a valid key into the reference database.
     *
     *  \return 0 on success, <0 on error, >0 on warning. errorString() returns descriptive text.
     */
    int operator()( CalibratedRadianceSpectrum output, 
                     const CalibratedRadianceSpectrum input, 
                     DetectorIndex_t pixel ); 
};

#endif

