/**
  Test program to demonstrate the core resampling algorithm in C++
  author: Maciek Smuga-Otto
  copyright 2004 University of Wisconsin-Madison.
*/
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>

#include <FBF.h>
#include <fftw3.h>

#include <iostream>
#include <vector>
#include <string>
#include <complex>
#include <functional>
#include <algorithm>


#define OUT_PREFIX "test2_"

// compile as follows:
//gcc -Wall -I/home/maciek/cvs/TOOLS/dev/librkg/fbf -I/home/maciek/opt/fftw-3.0.1/include -c fftw3zfli.cc
//g++ -o fftw3zfli fftw3zfli.o -L/home/maciek/opt/fftw-3.0.1/lib -lfftw3 -lstdc++
using namespace std;

static const size_t SPECTRUM_SIZE = 1025;
static const size_t IFG_SIZE = 2048;
static const size_t UPSAMPLED_IFG_SIZE = 65536;
static const size_t UPSAMPLED_SPECTRUM_SIZE = 32769;
static const float RENORM_RATIO = 1.0 / 2048;

class RFFTW_scratch {
  public:
    vector<double> spec_reflected;
    vector<double> short_ifg;
   
    vector<double> ifg_zerofilled;
    vector<double> zeros;
    vector<double> spec_upsampled_reflected;

    RFFTW_scratch(): spec_reflected( IFG_SIZE, 0.0 ),
                     short_ifg( IFG_SIZE, 0.0 ), 
                     ifg_zerofilled( UPSAMPLED_IFG_SIZE, 0.0 ),
                     zeros( UPSAMPLED_IFG_SIZE, 0.0 ), 
                     spec_upsampled_reflected( UPSAMPLED_IFG_SIZE, 0.0 )   {}
    ~RFFTW_scratch();
};

int upsample( vector<double> &spec_upsampled, const vector<double> &spec_in, 
              const fftw_plan planforth, const fftw_plan planback, 
              RFFTW_scratch *scratch )
{
    // reflect the spectrum
    copy( spec_in.begin(), spec_in.end(), scratch->spec_reflected.begin() );
    copy( spec_in.begin() + 1, spec_in.end(), scratch->spec_reflected.rbegin() );

    // transform to halfcomplex interferogram
    fftw_execute( planforth );

    // copy ends to larger halfcomplex interferogram vector.
    copy( scratch->zeros.begin(), scratch->zeros.end(), scratch->ifg_zerofilled.begin() );
    copy( scratch->short_ifg.begin(), scratch->short_ifg.begin() + SPECTRUM_SIZE, 
          scratch->ifg_zerofilled.begin() );
    copy( scratch->short_ifg.rbegin(), scratch->short_ifg.rbegin() + SPECTRUM_SIZE - 2, 
          scratch->ifg_zerofilled.rbegin() );
    scratch->ifg_zerofilled[SPECTRUM_SIZE-1] *= 0.5; // hack to overcome a possible fftw bug.
    
    // re-transform to upsampled spectrum
    fftw_execute( planback );
    
    transform( scratch->spec_upsampled_reflected.begin(), 
               scratch->spec_upsampled_reflected.begin() + UPSAMPLED_SPECTRUM_SIZE,
               spec_upsampled.begin(),
               bind2nd(multiplies<double>(), RENORM_RATIO) );

    return 0;
}

int linear_downsample( vector<double> &spec_out, const vector<double> &spec_upsampled, 
                       const double vlaser_ratio )
{
    double A = 32.0 * ( SPECTRUM_SIZE - 1 ) * ( 1 / vlaser_ratio - 1 );
    double B = 32.0 / vlaser_ratio;
    double r, rf; // fractional resample position
    int ri; // integral resample position
    for (unsigned int n = 0; n < SPECTRUM_SIZE; ++n )
    {
        r = A + B * n;
        ri = (int)r;
        rf = r - ri;

        if (r < 0 || r >= UPSAMPLED_SPECTRUM_SIZE)
            spec_out[n] = 0;
        else
            spec_out[n] = spec_upsampled[ri] 
                        + (spec_upsampled[ri+1] - spec_upsampled[ri]) * rf;
    }
    return 0;
}

int main()
{
    vector<double> spec_in( SPECTRUM_SIZE, 0.0 );
    vector<double> spec_upsampled( UPSAMPLED_SPECTRUM_SIZE, 0.0 );
    vector<double> spec_out( SPECTRUM_SIZE, 0.0 );

    RFFTW_scratch *scratch = new RFFTW_scratch();

    // FFT plans // flags
    fftw_plan planforth = fftw_plan_r2r_1d( IFG_SIZE,
                                            (double *)&scratch->spec_reflected[0],
                                            (double *)&scratch->short_ifg[0], 
                                            FFTW_R2HC, 
                                            FFTW_ESTIMATE
                                           );
    fftw_plan planback = fftw_plan_r2r_1d( UPSAMPLED_IFG_SIZE,
                                           (double *)&scratch->ifg_zerofilled[0],
                                           (double *)&scratch->spec_upsampled_reflected[0], 
                                           FFTW_HC2R, 
                                           FFTW_ESTIMATE
                                          );

    const double vlaser_ratio = 0.999;
    const string spec_in_filename("maxradLW_rolloff600-1100.real8.1025");

    // load input spectrum file file
    FILE *fspec_in = fopen( spec_in_filename.c_str(), "rb" );
    FBF::load( spec_in, fspec_in, true, SPECTRUM_SIZE );   
    fclose( fspec_in );

    upsample( spec_upsampled, spec_in, planforth, planback, scratch );
    FBF::save( OUT_PREFIX "fftw3_upsampled.real8.32769", spec_upsampled );

    linear_downsample( spec_out, spec_upsampled, vlaser_ratio );
    FBF::save( OUT_PREFIX "fftw3_resampled.real8.1025", spec_out );

    fftw_destroy_plan( planforth );
    fftw_destroy_plan( planback );
}

    
