#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <FBF.h>

#include <iostream>
#include <vector>
#include <string>
#include <complex>
#include <functional>
#include <algorithm>

#include <fftw.h>
#include <rfftw.h>

#define OUT_PREFIX "test2_"

// compile as follows:
// gcc -Wall -I/home/maciek/cvs/TOOLS/dev/librkg/fbf -lrfftw -lfftw -lm -lstdc++ -o rfftw2zfli rfftw2zfli.cc 

using namespace std;

static const size_t SPECTRUM_SIZE = 1025;
static const size_t IFG_SIZE = 2048;
static const size_t UPSAMPLED_IFG_SIZE = 65536;
static const size_t UPSAMPLED_SPECTRUM_SIZE = 32769;
static const float RENORM_RATIO = 1.0 / 2048;

class RFFTW_scratch {
  public:
    vector<double> spec_reflected;
    vector<double> short_ifg;
   
    vector<double> ifg_zerofilled;
    vector<double> zeros;
    vector<double> spec_upsampled_reflected;

    RFFTW_scratch(): spec_reflected( IFG_SIZE, 0.0 ),
                     short_ifg( IFG_SIZE, 0.0 ), 
                     ifg_zerofilled( UPSAMPLED_IFG_SIZE, 0.0 ),
                     zeros( UPSAMPLED_IFG_SIZE, 0.0 ), 
                     spec_upsampled_reflected( UPSAMPLED_IFG_SIZE, 0.0 )   {}
    ~RFFTW_scratch();
};

int upsample( vector<double> &spec_upsampled, const vector<double> &spec_in, 
              const rfftw_plan planforth, const rfftw_plan planback, 
              RFFTW_scratch *scratch )
{
    // reflect the spectrum
    copy( spec_in.begin(), spec_in.end(), scratch->spec_reflected.begin() );
    copy( spec_in.begin() + 1, spec_in.end(), scratch->spec_reflected.rbegin() );

    // transform to halfcomplex interferogram
    rfftw_one( planforth, (fftw_real *)&scratch->spec_reflected[0], 
                          (fftw_real *)&scratch->short_ifg[0] );

    // copy ends to larger halfcomplex interferogram vector.
    copy( scratch->zeros.begin(), scratch->zeros.end(), scratch->ifg_zerofilled.begin() );
    copy( scratch->short_ifg.begin(), scratch->short_ifg.begin() + SPECTRUM_SIZE, 
          scratch->ifg_zerofilled.begin() );
    copy( scratch->short_ifg.rbegin(), scratch->short_ifg.rbegin() + SPECTRUM_SIZE - 2, 
          scratch->ifg_zerofilled.rbegin() );
    scratch->ifg_zerofilled[SPECTRUM_SIZE-1] *= 0.5; // hack to overcome a possible rfftw bug.
    
    // re-transform to upsampled spectrum
    rfftw_one( planback, (fftw_real *)&scratch->ifg_zerofilled[0], 
                         (fftw_real *)&scratch->spec_upsampled_reflected[0] );
    
    transform( scratch->spec_upsampled_reflected.begin(), 
               scratch->spec_upsampled_reflected.begin() + UPSAMPLED_SPECTRUM_SIZE,
               spec_upsampled.begin(),
               bind2nd(multiplies<double>(), RENORM_RATIO) );

    return 0;
}

int linear_downsample( vector<double> &spec_out, const vector<double> &spec_upsampled, 
                       const double vlaser_ratio )
{
    double A = 32.0 * ( SPECTRUM_SIZE - 1 ) * ( 1 / vlaser_ratio - 1 );
    double B = 32.0 / vlaser_ratio;
    double r, rf; // fractional resample position
    int ri; // integral resample position
    for (unsigned int n = 0; n < SPECTRUM_SIZE; ++n )
    {
        r = A + B * n;
        ri = (int)r;
        rf = r - ri;

        if (r < 0 || r >= UPSAMPLED_SPECTRUM_SIZE)
            spec_out[n] = 0;
        else
            spec_out[n] = spec_upsampled[ri] 
                        + (spec_upsampled[ri+1] - spec_upsampled[ri]) * rf;
    }
    return 0;
}

int main()
{
    vector<double> spec_in( SPECTRUM_SIZE, 0.0 );
    vector<double> spec_upsampled( UPSAMPLED_SPECTRUM_SIZE, 0.0 );
    vector<double> spec_out( SPECTRUM_SIZE, 0.0 );

    RFFTW_scratch *scratch = new RFFTW_scratch();

    // FFT plans // flags
    rfftw_plan planforth = rfftw_create_plan( IFG_SIZE, 
                                            FFTW_REAL_TO_COMPLEX, 
                                            FFTW_ESTIMATE
                                           );
    rfftw_plan planback = rfftw_create_plan( UPSAMPLED_IFG_SIZE, 
                                           FFTW_COMPLEX_TO_REAL, 
                                           FFTW_ESTIMATE
                                          );

    const double vlaser_ratio = 0.9;
    const string spec_in_filename("maxradLW_rolloff600-1100.real8.1025");

    // load input spectrum file file
    FILE *fspec_in = fopen( spec_in_filename.c_str(), "rb" );
    FBF::load( spec_in, fspec_in, true, SPECTRUM_SIZE );   
    fclose( fspec_in );

    upsample( spec_upsampled, spec_in, planforth, planback, scratch );
    FBF::save( OUT_PREFIX "rfftw_upsampled.real8.32769", spec_upsampled );

    linear_downsample( spec_out, spec_upsampled, vlaser_ratio );
    FBF::save( OUT_PREFIX "rfftw_resampled_0-9.real8.1025", spec_out );
}

    
