static const char _cvsid[] =
"$Id: ResamplingStage.cc,v 1.11 2004/06/18 20:23:25 maciek Exp $";
static const char _copyright[] = 
"Copyright 2004, University of Wisconsin Space Science & Engr. Center."; 


/** 
*  \file ResamplingStage.cc
*  \brief Implementation of a processing stage executable using SincResampler operator.
*
*  This stage implements a simplified standalone executable processing stage using
*  memory mapped data channel, and piped audit and monitoring channel.
*  The stage logic is responsible for negotiating with the framework code to make 
*  sure that the data structures being provided and delivered conform to the needs 
*  of the enclosed operators. 
*  The stage logic also accepts control information forwarded from the pipeline 
*  sequencing logic by the framework software.
*  This module may be ultimately superseded by an equivalently testable module 
*  for a more advanced middleware solution (e.g. ACE). 
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id $
*
*  \par Revision Log:
*  \verbatim
*
*  $Log: ResamplingStage.cc,v $
*  Revision 1.11  2004/06/18 20:23:25  maciek
*  changes to permit operating mode where cache of FMatrices is flushed regularly
*  (in this case, after each call to resample), so as to permit running the code
*  on a full cube, on a single machine, without overflowing memory resources.
*
*  Revision 1.10  2004/03/29 22:58:57  rayg
*  Merged changes in code from SinglePrecision_RKG_200403 branch.
*  Build with USE_SINGLE_PRECISION env variable set to enable single prec mode.
*
*  Revision 1.9.2.1  2004/03/19 21:51:07  rayg
*  Single precision test version
*  modifications to makefile, errortest1, all sources
*
*  Revision 1.9  2004/03/16 16:31:49  rayg
*  Profiling of F matrix generation and resampling operation.
*  If used, requires libclok.a from TOOLS/dev/librkg/time/
*
*  Revision 1.8  2004/02/21 21:52:38  rayg
*  Forgot eoln on ident response.
*
*  Revision 1.7  2004/02/21 21:49:29  rayg
*  Fix for ATLAS doing C++-style mangling on cblas.h
*  Addition of ident command.
*
*  Revision 1.6  2004/02/21 21:19:31  rayg
*  added 'ident' command and _cvsid
*
*  Revision 1.5  2004/02/20 16:10:39  rayg
*  Simple testing harness simpletest0.py runs to completion. Next, create
*  simpletest1.py which uses regression testing data.
*
*  Revision 1.4  2004/02/20 05:00:11  rayg
*  Modified back to passing file descriptors - memory maps are not necessarily
*  treated as physical memory addresses by things like Java and Python.
*
*  Revision 1.3  2004/02/20 04:09:53  rayg
*  Approaching readiness for verification against reference implementation.
*  Still need a quick python execution wrapper for testing which will
*  memory map appropriate flat binary reference test data files.
*  Changed input environment variables significantly in the simplifying direction.
*
*  Revision 1.2  2004/02/19 18:01:31  rayg
*  This version is about 80% complete. It needs to be provided information on the dimensionality
*  of the inputs (flat binary convention for starters) and it needs to properly initialize
*  the adapters which are used by the operator..
*
*  Revision 1.1  2004/02/19 17:02:29  rayg
*  Main routine for a resampling stage using memory-mapped files for framework
*  communication and the SincResampler operator for the algorithm implementation.
*
*
*  \endverbatim
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
*  \endverbatim
*/



#include "SincResampler.h"




/* C includes */
#include <cassert>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>

/* STL includes */
#include <map>
#include <string>
#include <strstream>

using namespace blitz;

/** 
* Globals. 
*/


/* file descriptors inherited from the parent process */
int fdInput = -1;
int fdOutput = -1;
int fdDatabase = -1;


/* and the corresponding memory mapped buffers */
/* const */ data_t *mmInput = NULL;
data_t *mmOutput = NULL;
ResamplingSettings_t *mmDatabase = NULL;

/** size in bytes of records in input map */
size_t inputRecordSize = 0;
/** record size for output map (should be identical) */
size_t outputRecordSize = 0; 
/** record size for reference data is sizeof(ResamplingSettings_t) */
size_t databaseRecordSize = sizeof( ResamplingSettings_t );

/** number of available input buffers */
size_t inputRecordCount = 0;
/** number of available output buffers */
size_t outputRecordCount = 0;
/** number of available reference setting structures */
size_t databaseRecordCount = 0;


/** Attach buffers needed to interface to the framework
 * These are specified in environment variables.
 * varInputSpectra: "%d,%u,%u" herited read-only memory map to be made from of file descriptor (%d) containing (%u) floats for each of (%u) buffers; buffers are indexed from 0 (i.e. offset)
 * varOutputSpectra: "%d,%u,%u" herited read-write memory map to be made from of file descriptor (%d) containing (%u) floats for each of (%u) buffers; buffers are indexed from 0 (i.e. offset)
 * refSamplingSettings: "%d,%u" heritable read-only fd (%d) to be used memory map containing (%d) ResamplingSettings_t structures
 */
void attachBuffers( void )
{
    const char *env; 
    int rc;
    
    env = getenv( "varInputSpectra" );
    assert( env /* varInputSpectra not set */ );
    rc = sscanf( env, "%d,%u,%u", &fdInput, &inputRecordSize, &inputRecordCount );
    assert( rc==3 /* varInputSpectra did not scan correctly */ );
    mmInput = (data_t *)mmap( NULL, inputRecordSize*inputRecordCount*sizeof(data_t), 
                                    PROT_READ, MAP_SHARED, fdInput, 0 );
    assert( mmInput );
    
    env = getenv( "varOutputSpectra" );
    assert( env /* varOutputSpectra not set */ );
    rc = sscanf( env, "%d,%u,%u", &fdOutput, &outputRecordSize, &outputRecordCount );
    assert( rc==3 /* varOutputSpectra did not scan correctly */ );
    mmOutput = (data_t *)mmap( NULL, outputRecordSize*outputRecordCount*sizeof(data_t), 
                                     PROT_READ | PROT_WRITE, MAP_SHARED, fdOutput, 0 );
    assert( mmOutput );
    
    env = getenv( "refSamplingSettings" );
    assert( env /* refSamplingSettings not set */ );
    rc = sscanf( env, "%d,%u", &fdDatabase, &databaseRecordCount );
    assert( rc==2 /* refSamplingSettings did not scan correctly */ );
    mmDatabase = (ResamplingSettings_t *)mmap( NULL, sizeof(ResamplingSettings_t)*databaseRecordCount, PROT_READ, MAP_SHARED, fdDatabase, 0 );
    assert( mmInput );
}

/** Detach connection to driver/framework.
 */
void detachBuffers( void ) 
{
    int rc = msync( mmOutput, outputRecordSize*outputRecordCount*sizeof(data_t), MS_SYNC );
    assert(rc==0);
    rc = munmap( (void *)mmOutput, outputRecordSize*outputRecordCount*sizeof(data_t) );
    assert(rc==0);
    rc = munmap( (void *)mmInput, inputRecordSize*inputRecordCount*sizeof(data_t) );
    assert(rc==0);
    rc = munmap( (void *)mmDatabase, sizeof(ResamplingSettings_t)*databaseRecordCount ); 
    assert(rc==0);    
}


/** Monitoring service adapter. 
 * Implements monitoring service for the resampling operator in terms of 
 * talking to the monitoring channel provided by the framework.
 * Future version of this should follow an XML-like standard for exchanging monitoring events with framework.
 */
class MonitorAdapter: public MonitoringServiceForSincResampler
{
    /** file descriptor representing the link to the framework monitor.
    */
    int fd; 
#ifdef PROFILING
    double TFwall, TFuser, TFsys, TRwall, TRuser, TRsys;
    unsigned NF, NR;
#endif
public:

    MonitorAdapter( int _fd ): fd(_fd)
#ifdef PROFILING
        , TFwall(0.), TFuser(0.), TFsys(0.), TRwall(0.), TRuser(0.), TRsys(0.), NF(0), NR(0) 
#endif
    { }

    
    inline void write( const std::ostringstream &B ) 
    {
        std::string S(B.str());
        ::write( fd, S.c_str(), S.size() );
    }
    
    virtual void resamplingCompleted()
    {
        std::ostringstream B;
        B << "resampling completed" << endl;
        write(B);
    }

    virtual void resamplingAborted()
    {
        std::ostringstream B;
        B << "resampling aborted" << endl;
        write(B);
    }

    virtual void addingEntryToResamplingCache( DetectorIndex_t forThisDetector, size_t resultantCacheSize )
    {
        std::ostringstream B;
        B << "adding entry for detector " << forThisDetector << " to resampling cache" << endl;
        write(B);
    }

    virtual void removingEntryFromResamplingCache( DetectorIndex_t forThisDetector, size_t resultantCacheSize )
    {
        std::ostringstream B;
        B << "removing entry for detector " << forThisDetector << " to resampling cache" << endl;
        write(B);        
    }
    
    virtual void resamplingCacheFlushed( )
    {
        std::ostringstream B;
        B << "resampling cache flushed" << endl;
        write(B);        
    }
    
    virtual void referenceResamplingSettingsNotFound( DetectorIndex_t forThisDetector )
    {
        std::ostringstream B;
        B << "ERROR - unable to find resampling settings for " << forThisDetector << endl;
        write(B);        
    }
    
    virtual void timeCostOfFMatrix( double wall, double user, double system )
    {
#ifdef PROFILING
        TFwall+= wall; TFuser+=user; TFsys+=system; NF ++; 
        std::ostringstream B;
        B << ">>> tFMatrixWall,tFMatrixUser,tFMatrixSystem = " << wall << ',' << user << ',' << system << endl;
        write(B);
#endif        
    }
    
    virtual void timeCostOfResampling( double wall, double user, double system )
    {
#ifdef PROFILING
        TRwall+= wall; TRuser+=user; TRsys+=system; NR ++;
        std::ostringstream B;
        B << ">>> tResampleWall,tResampleUser,tResampleSystem = " << wall << ',' << user << ',' << system << endl;
        write(B);
#endif
    }

    ~MonitorAdapter( )
    {
#ifdef PROFILING
        std::ostringstream B;
        B << ">>> nFMatrix,tTotalFMatrixWall,tTotalFMatrixUser,tTotalFMatrixSystem = " << NF << ',' << TFwall << ',' << TFuser << ',' << TFsys << endl;
        B << ">>> nResample,tTotalResampleWall,tTotalResampleUser,tTotalResampleSystem = " << NR << ',' << TRwall << ',' << TRuser << ',' << TRsys << endl;
        write(B);
#endif        
    }
};


/** Auditing adapter to framework
 */
class AuditAdapter: public AuditingServiceForSincResampler
{
    
};


/** Reference database adapter.
 * provides reference data structures to sinc resampler operator.
 * This implementation takes in a flat memory array with the key as the offset into the array.
 */

class ReferenceDatabaseAdapter: public ReferenceDatabaseForSincResampler 
{
    const ResamplingSettings_t *map;
    size_t mapRecords;
public:
    
    ReferenceDatabaseAdapter( const ResamplingSettings_t *_map, size_t _max_elements ): 
        map(_map), mapRecords( _max_elements )
    {
    }
    
    virtual const ResamplingSettings_t *operator[]( DetectorIndex_t key )
    {
        if (key>=mapRecords) return NULL;            
        return &map[(unsigned)key];
    }

};



/** Resample command implementation
 *  
 * Given an input buffer number and an output buffer number,
 * Find the F matrix that we need in the cache. 
 *  If it's not in the cache, add it to the cache.
 *   If the cache is too big, flush out a random entry or the LRU entry
 * Multiply the input spectrum by the corresponding F-matrix
 * Rrite the output to the output buffer
 * Return control and status to the command loop for it to respond to 
 *   the managing (parent) application.
 */
int Resample( SincResampler &R, int dexout, int dexin, int refin )
{
    assert( refin < databaseRecordCount );

    /* const */ data_t *input_data = mmInput + dexin*inputRecordSize;
    assert( dexin < inputRecordCount );
    
    data_t *output_data = mmOutput + dexout*outputRecordSize;
    assert( dexout < outputRecordCount );
    
    assert( inputRecordSize==outputRecordSize /* precondition of algorithm as implemented */ );

    /* provide blitz interfaces to memory blocks of interest */
    TinyVector<int,1> iwidth(inputRecordSize);
    /* const */ CalibratedRadianceSpectrum X( input_data, iwidth, neverDeleteData );
    TinyVector<int,1> owidth(outputRecordSize);
    CalibratedRadianceSpectrum Y( output_data, owidth, neverDeleteData );
    
    return R( Y, X, (DetectorIndex_t)refin );
}




/**
 * main command loop. 
 *  at the moment, quick and dirty. 
 *  later, toolbox approach to doing these kinds of things to impose some uniformity across apps.
 *  one line input -> one line output. 
 * control channel is typically stdin and stdout
 */
void cmdLoop( FILE *out, FILE *in )
{
    ReferenceDatabaseAdapter db( mmDatabase, databaseRecordCount );
    MonitorAdapter monitor( 2 ); // FIXME: for now, this goes to stderr
    AuditAdapter audit;
    
    SincResampler resampler( db, audit, monitor );
    
    char cmd[ 256 ];
    int p1, p2, p3, rc; 
    while( !feof( in ) )
    {
        cmd[0] = 0;
        fgets( cmd, sizeof(cmd)-1, in );
        
        if (strstr(cmd,"exit")==cmd)
        {
            fprintf( out, "0 OK, exiting\n" );
            break;
        }
        
        if (strstr(cmd,"ping")==cmd)
        {
            fprintf( out, "0 OK, ready for command\n" );
            continue;
        }
        
        if (strstr(cmd,"ident")==cmd)
        {
            fprintf( out, "0 OK, %s\n", _cvsid );
            continue;
        }
        
        if (strstr(cmd,"flush")==cmd)
        {
            resampler.referenceDatabaseHasChanged();
            fprintf( out, "0 OK, FMatrix cache flushed\n" );
            continue;
        }
        
        /* resample command: given index into settings database, and index into
            input and output maps, execute a resampling.
            At a later time, we may want to consider formatting the command channel as SOAP/XML, 
            provided we don't lose a disproportionate number of CPU cycles breaking down such a format! 
            */
        if (3==sscanf( cmd, "resample %d %d %d", &p1, &p2, &p3 ))
        {
            rc = Resample( resampler, p1, p2, p3 );
            fprintf( out, "%d %s\n", rc, rc==0? "OK" : rc<0? "ERROR" : "WARNING" );
            /* report should optionally include a descriptive text trailer, or be XML formatted */
            continue;
        }
        
        /* flush command: indicates that the reference database has changed */
        
        /* ident command: requests identification information on the software and algorithm version */
        
        fprintf( out, "? unknown command\n" );
    }
}


#ifndef TEST

int main( int argc, char **argv )
{    
    /* change stdin/stdout to line-mode buffering for accepting commands */
    setvbuf( stdin, NULL, _IOLBF, 0 );
    setvbuf( stdout, NULL, _IOLBF, 0 );
    
    attachBuffers();
    cmdLoop( stdout, stdin );
    detachBuffers();
}

#endif




