/** 
*  \file SincResampler.h
*  \brief Interface specification for Spectral Resampling Algorithm.
*    This is an implementation of a "sinc resampling" operator per 
*    the matrix multiplication approach as defined in the 
*    BOMEM technical memo BOM-TECH-XXXX(Draft)-Spectral Resampling(work-02).doc.
*
*    This algorithm implementation is intended to be compatible with
*    GIFTS real-time performance concerns while maintaining compatibility
*    with testing harnesses and alternate framework implementations.
*
*    This operator requires a set of abstracted services which provide 
*    monitoring, auditing and reference database functionality, which
*    is deliberately out of scope of this interface spec. In practice,
*    this operator will be embedded into a larger processing stage which
*    implements these interfaces. 
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: SincResampler.h,v 1.4 2004/03/29 22:58:57 rayg Exp $
*
*  \par Revision Log:
*  \verbatim
*
*  $Log: SincResampler.h,v $
*  Revision 1.4  2004/03/29 22:58:57  rayg
*  Merged changes in code from SinglePrecision_RKG_200403 branch.
*  Build with USE_SINGLE_PRECISION env variable set to enable single prec mode.
*
*  Revision 1.3.2.1  2004/03/19 21:51:07  rayg
*  Single precision test version
*  modifications to makefile, errortest1, all sources
*
*  Revision 1.3  2004/03/16 16:31:49  rayg
*  Profiling of F matrix generation and resampling operation.
*  If used, requires libclok.a from TOOLS/dev/librkg/time/
*
*  Revision 1.2  2004/02/19 15:47:09  rayg
*  New low-level operator (as opposed to stage) interface with abstract C++ interfaces to services.
*  Removed imaginary part of incoming spectra, as current use of resampling algorithms does not require it nor use
*  it as a quality metric.
*  Unit testing not yet completed.
*
*  Revision 1.1  2004/02/18 16:19:58  rayg
*  proposed operator interface
*
*
*  \endverbatim
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
*  \endverbatim
*/

#ifndef H_SincResampler
#define H_SincResampler

#ifndef NO_DEPENDENCY_INCLUDES
/* Dependent headers for the interface.
 */

/* system includes */
#include <string>

/* Third party includes */
#include <blitz/array.h>

/* Internal includes */
// #include "common.h"

#endif

/*
 * The following typedefs get moved to a common.h declaration file linking the underlying 
 * array/matrix representations to named types appropriate to the domain. 
 */

/* data type for elements of input and output spectra */
#ifdef USE_SINGLE_PRECISION
    typedef float data_t; 
#else
    typedef double data_t;
#endif

/** A calibrated radiance spectrum with any imaginary component discarded. 
 * Units: mW/(m2.sr.cm-1)
 */
typedef blitz::Array<data_t,1> CalibratedRadianceSpectrum; 

/** Detector Index. 
 * Effectively a key into the reference database. 
 * Expect that this will be used for caching of resampling matrices.
 * Units: none.
 */
typedef unsigned long DetectorIndex_t;





/** 
 * \struct Resampling Settings structure
 * Reference Database structure for a detector used by SincResampler.
 */
struct ResamplingSettings_t
/* is native byte order, IEEE floating point and integer */
/* Put this in a header file. And an interface control document. */
{ 
    /** source laser wavenumber -- varies per off-axis angle 
     * Units: cm-1
     */
    double sourceLaserWavenumber;
    /** target laser wavenumber -- generally constant 
     * Units: cm-1
     */
    double targetLaserWavenumber;
    /** number of points in the input spectrum -- e.g. 1025 for a 2048 interferogram 
     * Units: 
     */
    unsigned long inputPointCount; 
    /** quality reduction factor reduces the effective size of the resampling matrix at cost of error. 
     * Units:
     */
    unsigned long qualityReductionFactor;  // not yet used    
    /** decimation factor 
     * Units: 
     */
    unsigned long decimationFactor;  // not yet used 
};



/** Sinc Resampler abstract interface to Auditing Service
 */
class AuditingServiceForSincResampler
{
public:

    
};


/** \class MonitoringServiceForSincResampler
 * \brief Sinc Resampler abstract interface to Monitoring Service
 */
class MonitoringServiceForSincResampler
{
public:
    /** completion of job
    */
    virtual void resamplingCompleted() = 0;
    
    /** failure of job
    */
    virtual void resamplingAborted() = 0;
    
    /** Indicate that expensive operation of generating a new resampling matrix is taking place.
    */
    virtual void addingEntryToResamplingCache( DetectorIndex_t forThisDetector, size_t resultantCacheSize ) = 0;
    
    /** Indicate that entry is being removed from the cache. 
    *   If this is happening consistently, we may be thrashing.
    */
    virtual void removingEntryFromResamplingCache( DetectorIndex_t forThisDetector, size_t resultantCacheSize ) = 0;
    /** Indicate that cache flush has occurred. 
    */
    virtual void resamplingCacheFlushed( ) = 0;
    /** Indicate that a reference database fetch returned no information
    */
    virtual void referenceResamplingSettingsNotFound( DetectorIndex_t forThisDetector ) = 0;
    
    /** if compiled with PROFILING flag, these will be reported after the completion of an operation
     */
    virtual void timeCostOfFMatrix( double wall, double user, double system ) = 0;
    virtual void timeCostOfResampling( double wall, double user, double system ) = 0;
};


/** \class ReferenceDatabaseForSincResampler
 * \brief Sinc Resampler abstract interface to Reference Database. 
 * Client interface to reference database used by Sinc Resampler to fetch ResamplingSettings_t structs.
 */
class ReferenceDatabaseForSincResampler 
{
public:

    /** Return reference database structure for a given detector, or NULL if none is available.
    */
    virtual const ResamplingSettings_t *operator[]( DetectorIndex_t ) = 0;
};


/** SincResampler operator.
 * Instances of this object use the provided services to maintain an internal cache 
 * of transformations in order to resample the calibrated radiance spectra provided.
 */
class SincResampler
{
private:
    struct SincResamplerInternals &my;
    
public:

    /** Constructor
    */
    SincResampler( ReferenceDatabaseForSincResampler &, 
                   AuditingServiceForSincResampler &, 
                   MonitoringServiceForSincResampler & );
    /** Destructor
    */
    ~SincResampler();
    
    /** Return a string describing an error code.
     */
    static const std::string &errorString(); 
    
    /** Invalidate cache contents. 
     * This informs the operator that the reference database has changed.
     */
    void referenceDatabaseHasChanged(); 
    
    /** Hint preferred max cache size in bytes. 
     *  May be rounded down to nearest convenient page size prior to use.
     */
    void setMaxCacheSize( unsigned );
    static const unsigned CACHE_SIZE_UNLIMITED; 
    
    /** Read max cache size in bytes.
    */
    unsigned maxCacheSize( ) const;
    
    /** Execute spectral resampling.
     * 
     *  \param output Output spectrum must be allocated identical in size to input but need not be initialized. 
     *  \param input Input spectrum corresponds to the sourceLaserWavenumber sampling stated in the resultant settings.
     *  \param pixel Detector Index must be a valid key into the reference database.
     *
     *  \return 0 on success, <0 on error, >0 on warning. errorString() returns descriptive text.
     */
    int operator()( CalibratedRadianceSpectrum &output, 
                     const CalibratedRadianceSpectrum &input, 
                     DetectorIndex_t pixel ); 
};

#endif

