"""
$Id: FBF.py,v 1.21 2004/03/04 18:34:57 maciek Exp $
Flat Binary Format python utilities
parallel to TOOLS/Mfiles/fbf*.m
"""

import os
import os.path
import sys
import string
import struct
import Numeric
import types

fbf_file_types = { 
    'char1': ( 'c', 0, 1 ),
    'char': ('c', 0, 1 ),
    'int1': ('b',0,1),
    'int2': ('h',0,2),
    'int4': ('l',0,4),
    'integer4': ('l',0,4),
    'real4': ('f',0,4),
    'real8': ('d',0,8),
    'complex8': ('f',1,8),
    'complex16': ('d',1,16)
}

fbf_endian = { 
    '>': string.upper,
    '<': string.lower,
    'big': string.upper,
    'little': string.lower
}

if not hasattr(sys,'byteorder'): 
    piffle = struct.pack( '=l', 0xabcd )
    if struct.pack( '<l', 0xabcd ) == piffle:   byteorder = 'little'
    elif struct.pack( '>l', 0xabcd ) == piffle: byteorder = 'big'
    else: 
        print >>sys.stderr, "Warning: Unable to identify byte order as 'big' or 'little'. Defaulting to 'big'."
        byteorder = 'big'
else:
    byteorder = sys.byteorder

fbf_endian[ 'native' ] = fbf_endian[ byteorder ]
fbf_endian[ '=' ] = fbf_endian[ byteorder ]


FBF_FMT = """< FBF object:
pathname: %(pathname)s
stemname: %(stemname)s
filename: %(filename)s
dirname : %(dirname)s
type    : %(type)s
grouping: %(grouping)s
byteorder: %(byteorder)s
array_type: %(array_type)s
element_size: %(element_size)s
record_size: %(record_size)s >
""" 


class FBF:
    def __init__( self, stemname=None, typename=None, grouping=[], dirname='.', byteorder='native' ): 
        if not typename: 
            if stemname: self.attach( stemname )
            else: return
        else:
            self.build( stemname, typename, grouping, dirname, byteorder )
        return

    def packing_format( S, N=1 ):
        """routine which returns the struct module format string representing a single record
           of the file.
        """
        if S.is_complex==1: return '%c%d%c' % ( S.endian, S.record_elements*2*N, S.array_type )
        else: return '%c%d%c' % ( S.endian, S.record_elements*N, S.array_type )

    def attach( S, pathname ):
        dn,fn = os.path.split( pathname )
        parts = string.split( fn, '.' )
        S.pathname = pathname
        S.stemname = parts[0]
        S.type = parts[1]
        S.grouping = map( lambda x: int(x), parts[2:] )
        if not S.grouping: S.grouping = [ 1 ]
        S.dirname = dn
        if not S.dirname: S.dirname = '.'
        S.filename = fn
        TYPE = string.upper(S.type)
        if S.type==TYPE : 
            S.byteorder='big'
            S.endian='>'
        else: 
            S.byteorder='little'
            S.endian='<'
        if (sfx_remap.has_key(TYPE)): S.type = sfx_remap[TYPE]
        atype,is_complex,bytes = fbf_file_types[ string.lower(S.type) ]
        S.is_complex = is_complex
        S.element_size = bytes
        S.record_elements = reduce( lambda x,y: x*y, S.grouping )
        S.record_size = bytes * S.record_elements
        S.array_type = atype
        # using slice-copy idiom due to Jython's incorrect copy-constructor behavior.
        l = list( S.grouping[:] ) 
        l.reverse()
        S.gnipuorg = tuple(l)
        return S

    def build( self, stemname, typename, grouping=[], dirname='.', byteorder='native' ): 
        """ build an FBF descriptor from scratch
            byteorder can be 'big', 'little', 'native'
        """
        typename = fbf_endian[byteorder]( typename )
        filename = '%s.%s' % (stemname, typename)
        if grouping: filename = '%s.%s' % ( filename, string.join( [ str(x) for x in grouping ], '.' ) )
        self.attach( os.path.join( dirname, filename ) ) 
        
    def __repr__( self ): 
        return FBF_FMT % vars(self)
    
    def fp(self,mode="rb"): 
        if hasattr(self,'file'): return self.file
        return open( self.pathname, mode )

    def open( self, mode="rb" ): 
        self.file = self.fp(mode)
        return self.file
        
    def create( self, mode="wb+" ):
        self.file = self.fp(mode)
        return self.file
        
    def close( self ):
        if hasattr( self,'file' ): 
            del self.file
            return 0
        return 1

    def length( self ):
        try: # file length is bad if we don't flush
            if self.file and self.pending_flush==1: 
                self.file.flush()
                self.pending_flush = 0
        except:
            pass
        flen = os.stat( self.pathname )[6] # fixme: in future use .st_size as of Py2.2
        return flen / self.record_size
        
    def peg( self, x ):
        """ change negative record numbers over to real record numbers
            x is a tuple or list of record numbers
        """
        nrex = self.length()
        ret = []
        for r in x: 
            if r<0: ret.append( nrex + 1 + r )
            else: ret.append( r )
        return ret





sfx_remap = { 
    'STA': 'int1'
}

def info( pathname ):
    return FBF( pathname )

def build( stemname, typename, grouping=[], dirname='.', byteorder='native' ): 
    return FBF( stemname, typename, grouping, dirname, byteorder )

def swizzle( fmt, cplx, grp, dat ):
    x = Numeric.array( struct.unpack( fmt, dat ) )
    if not cplx: 
        return Numeric.reshape( x, [int(g) for g in grp] )
    xx = Numeric.reshape( x, ( len(x)/2, 2 ) )
    return Numeric.reshape( xx[:,0] + xx[:,1]*1j, grp )
        
def read( inp, start=1, end=0 ):
    #print "%s" % type(inp)
    if type(inp) == types.StringType: 
        # print "Info: open files beforehand to increase performance"
        inp = info( inp )
    fp = inp.fp()
    if type(start)==types.IntType: 
        start,end = inp.peg( (start,end) ) 
        if end<=start: end=start
        nrex = end-start+1
        ntoread = inp.record_size * nrex
        fp.seek( inp.record_size * (start-1) )
        # swipe in the whole load of data, and byte order correct it
        fmt = inp.packing_format( nrex )
        #print "going to read %d bytes into fmt %s" % (ntoread,fmt)
        data = fp.read( ntoread )
        if len(data) != ntoread:
            raise EOFError, "Could not read %d bytes for records %d-%d." % (ntoread,start,end)
        pre = []
        if nrex>1: pre = [nrex,]
        return swizzle( fmt, inp.is_complex, pre + list(inp.gnipuorg), data )
    else: # list or array
        ret = []
        rf = inp.peg( start )
        fmt = inp.packing_format()
        for r in rf:
            fp.seek( inp.record_size * (r-1) )
            data = fp.read( inp.record_size )
            if len(data) != inp.record_size:
                raise EOFError, "Could not read %d bytes for record %d." % (inp.record_size, r)
            ret.append( swizzle( fmt, inp.is_complex, inp.gnipuorg, data ) ) 
        return ret

def extract_indices_to_file( inp, indices, out_file ):
    if type(inp) == types.StringType: 
        inp = info( inp )
    fp = inp.fp()
    rf = inp.peg( indices )
    fmt = inp.packing_format()
    fout = open( out_file, 'wb' )
    for r in rf:
        fp.seek( inp.record_size * (r-1) )
        data = fp.read( inp.record_size )
        if len(data) != inp.record_size:
             raise EOFError, "Could not read %d bytes for record %d." % (inp.record_size, r)
        fout.write( data )
    fout.close()

def write( inp, rec, data ): 
    """ write a Numeric data array to a FBF record
    """
    if type(inp) == types.StringType: 
        inp = info( inp )
    fp = inp.fp('wb+')
    oldshape = data.shape
    data.shape = (inp.record_elements,)
    # format it up for disk storage and shove it away
    fp.seek( inp.record_size * (rec-1) )
    fmt = inp.packing_format( )
    fp.write( apply( struct.pack, ( fmt, ) + tuple( data ) ) )
    data.shape = oldshape
    # we need to flush data before using length()
    inp.pending_flush = 1

def block_write( inp, start_record, data ):
    """ write multiple records from a single numeric array
        returns number of records written
    """
    if type(inp) == types.StringType: 
        # print "Info: file access is faster for files previously opened"
        inp = info( inp )
    try:
        fp = inp.fp('rb+')
    except:
        fp = inp.fp('wb+')
    fp.seek( inp.record_size * (start_record-1) );
    x = Numeric.reshape( data, (Numeric.size(data),) )
    records = len(x) / inp.record_elements
    if 0!= len(x) % inp.record_elements:
        print "Error: blockWrite needs input to be a multiple of record size" 
        return 0
    s = data.shape
    is_complex=0
    if len(s)==1:
        if type(data[0])==complex:
           is_complex=1
    elif s[1]==1:
        if type(data[0])==complex:
           is_complex=1
    else:
        if type(data[0][0])==complex:
           is_complex=1
    if is_complex:
        if inp.is_complex != 1: 
            print "Error: trying to write complex data to a non-complex filename"
            return 0
        x = Numeric.concatenate([x.real,x.imag])
        x = Numeric.reshape( x, (2,len(x)/2) )
        x = Numeric.transpose( x )
        x = Numeric.reshape( x, (len(x)*2,) )
    fp.write( apply( struct.pack, ( inp.packing_format( records ), ) + tuple( x ) ) )
    inp.pending_flush = 1
    return records
    
    

# fold these right into the class, so they can be used selflessly or selfishly
FBF.write = write
FBF.read = read
FBF.block_write = block_write
FBF.extract_indices_to_file = extract_indices_to_file 

def length( inp ):
    if type(inp) == types.StringType: 
        inp = info( inp )
    return inp.length()


def test1():  # quick and dirty test from e-mail
    from Numeric import array
    fi = build( 'foo', 'complex8', [40,3], byteorder='little' )
    print fi
    fi = info('PCR_C.real4.40.26')
    print fi
    write( 'foo.real4.3', 1, array( [1.,3.,5.] ) )
    fi = info( 'bar.real4.3' )
    fi.create()
    print fi.length()
    fi.write( 2, array( [1.,4.,9.] ) )
    print fi.length()
    print fi.read(1)
    print fi.read(1,-1)
    print read( fi, [1,2,1] )
    print fi.read( [-1,-2,1,2] )
    fi.close()
    print "done"

    
if __name__=='__main__':
    from sys import argv
    fi = info(argv[1])
    data = read( fi )
    print data
    print fi
