#!/usr/bin/env python

""" Wrapper library for running the resampling stage
$Id: resampleWrapper2.py,v 1.5 2004/06/18 20:23:25 maciek Exp $
Original concept: R.K.Garcia
Converted to wrapper library: Maciek Smuga-Otto
Copyright 2004, University of Wisconsin
"""

import sys, os, string
from time import time
from Numeric import *
import FBF

from mmap import *
from struct import *
from math import sin
from os import tempnam, fork, environ, execv, dup2, fdopen, pipe, waitpid, path

###

DEBUG = 0
NOCACHE = 1

#RESAMPLING_EXECUTABLE='/home/rayg/Software/Resampling/prerelease_source/ResamplingStage' #DEFAULT
RESAMPLING_EXECUTABLE='/home/maciek/cvs/devel/Calibrate/Resample/ResamplingStage' #TESTING
#RESAMPLING_EXECUTABLE='/home/maciek/cvs/devel/Calibrate/Resample/ResamplingDummyStage.py' #DEBUG
RESAMPLING_EXECUTABLE_ZERO_THRESHOLD_VERSION = '/home/rayg/Software/Resampling/ResamplingStage_WithPerfEnhIndex'
RESAMPLING_EXECUTABLE_SINGLE_PRECISION_VERSION = '/home/rayg/Software/Resampling/ResamplingStage_SinglePrecision'

GIFTS_VLASER_OUT = 11737.0 #FIXME - this is almost certainly incorrect but irrelevant to this test.

GIFTS_IFG_SIZE = { 'LW': 2048, 'SMW': 4096 }
GIFTS_DECIMATION_FACTOR = { 'LW': 10, 'SMW': 5 }

### Communication with executable

def invoke( (fbf_out, fbf_in, fpref, reflen), exe ):
    rfd,wfd = pipe()
    writable0 = fdopen( wfd, 'w' )
    readable0 = fdopen( rfd, 'r' )
    rfd,wfd = pipe()
    writable1 = fdopen( wfd, 'w' )
    readable1 = fdopen( rfd, 'r' )
    
    pid = fork( )
    if (pid==0): # then I am the child
        nenv = { 'varInputSpectra': '%d,%d,%d' % \
                 (fbf_in.fp().fileno(), fbf_in.grouping[0], fbf_in.length()), 
                 'varOutputSpectra': '%d,%d,%d' % \
                 (fbf_out.fp().fileno(), fbf_out.grouping[0], fbf_out.length()),
                 'refSamplingSettings': '%d,%d' % (fpref.fileno(), reflen) }
        environ.update( nenv )
        writable0.close()
        readable1.close()
        print "child executing %s" % exe
        # for e in environ: print "%s: %s" % (e, environ[e]) # DEBUG

        # change stdin and stdout
        dup2( readable0.fileno(), 0 ) #stdin
        dup2( writable1.fileno(), 1 ) #stdout
        execv( exe, [ exe ] )
        # end .. 
        
    readable0.close()
    writable1.close()
    return writable0, readable1, pid
    
def command( (writable0,readable1,pid), cmd ):
    print >>writable0, cmd
    writable0.flush()
    return readable1.readline()
    
def ping( link ):
    return command( link, "ping" )
        
def ident( link ):
    return command( link, "ident" )
        
def flush( link ):
    return command( link, "flush" )

def resample( link, out_idx, in_idx, ref_idx ):
    # print >> sys.stderr, 'resample %s %s %s' % (out_idx, in_idx, ref_idx) #DEBUG
    return command( link, 'resample %s %s %s' % (out_idx, in_idx, ref_idx) )

def doexit( link ):
    return command( link, 'exit' )


### private utility methods

def prep_fbf_input( fbf_in ):
    fbf_in.open() # open FBF file in read mode

def prep_fbf_output( fbf_out, length ):
    fbf_out.create()
    fbf_out.block_write( 1, zeros([length,] + fbf_out.grouping, 'd') )
    fbf_out.fp().seek(0)

SINGLE_PRECISION = 'real4'
DOUBLE_PRECISION = 'real8'
def get_out_name( in_fbf, vlaser_string, out_label = 'out', algorithm = 'ccnote', precision = DOUBLE_PRECISION, zero_threshold_param = 0 ):
    vr_name = '_vr%s' % string.join( vlaser_string.split('.'), ',' )
    zep_name = ''
    if zero_threshold_param:
         zep_name = '_ztp%s' % zero_threshold_param
    grouping_name = string.join(['%s'% gg for gg in in_fbf.grouping], '.')
    filename = '%s_%s_%s%s%s.%s.%s' % ( out_label, algorithm, in_fbf.stemname, vr_name, zep_name, precision, grouping_name )
    if in_fbf.dirname:
        filename = os.path.join( in_fbf.dirname, filename )
    return filename
    
    
### Reference channel setup - returns a file pointer and number of records
def createGIFTSfpref( detector, vlaser_ratio = None, vlaser_in = None, vlaser_out =  GIFTS_VLASER_OUT, OPT1 = 0 ):
    """ vlaser_in is a float or array corresponding to the input vector of spectra, can also be specified by vlaser_ratio.
    vlaser_out is assumed to be the GIFTS default.  Values are assumed to be either
    Floats or Numeric arrays, thus permitting per-element overloaded operations
    OPT1 is the zero-thresholding optimization parameter (default 0)"""
        
    if not vlaser_in: # vlaser_in trumps vlaser_ratio. 
        if not vlaser_ratio:
            return None # Error - must specify one or the other.
        vlaser_in = vlaser_out / vlaser_ratio 

    if type( vlaser_in ) == type( 0.0 ):
        vlaser_in = [vlaser_in, ]

    if type(vlaser_out) == type(0.0):
        vlaser_out = ones( len(vlaser_in) ) * vlaser_out
    OPT2 = 0 # unused optimization factor.
    if DEBUG: print "vlaser in: %s\n vlaser out: %s" % (vlaser_in, vlaser_out) #DEBUG
        
    # create temporary file, fill it with vlasery goodness.
    fpref = file( tempnam(), 'wb+' )
    for vi, vo in zip( vlaser_in, vlaser_out ):
        try:
            vi = vi[0]
        except:
            pass
        refpacket = ( vo, vi, GIFTS_IFG_SIZE[ detector ] / 2 + 1, OPT1, OPT2 ) #FIXME - but why?
        if DEBUG: print refpacket # DEBUG
        fpref.write( pack( 'ddLLL', *refpacket ) )
        
    fpref.seek(0)
    return fpref, len( vlaser_in )
    

### Library interface (public API)
def gifts_main(fbf_in, fbf_out, detector, vlaser_out = GIFTS_VLASER_OUT, vlaser_in = None, vlaser_ratio = None, resampler = RESAMPLING_EXECUTABLE, zero_threshold_param = 0, num_epochs = 1, nocache = None ):
    fref_ptr, reflen = createGIFTSfpref( detector, vlaser_out = vlaser_out, vlaser_ratio = vlaser_ratio, OPT1 = zero_threshold_param ) 
    if zero_threshold_param:
        resampler = RESAMPLING_EXECUTABLE_ZERO_THRESHOLD_VERSION
    wrp = invoke( (fbf_out, fbf_in, fref_ptr, reflen), resampler )
    print "ping result: %s" % ping( wrp ).strip() #DEBUG
    print "ident result: %s" % ident( wrp ).strip() #DEBUG

    epoch_timings = range( num_epochs )
    for epoch in epoch_timings:
        if DEBUG: print "epoch %s" % epoch # DEBUG

        epoch_start = time()
        for idx in range( fbf_in.length() ):
            # print "ping result: %s" % ping( wrp ).strip() # DEBUG
            res = resample( wrp, idx, idx, idx )
            if NOCACHE:
                flush( wrp )
            if DEBUG: print "resample result: %s" % res # DEBUG
        epoch_timings[ epoch ] = ( epoch, time() - epoch_start )

    print "exit requesting result: %s" % doexit( wrp )
    print "waiting for exit.."
    waitpid( wrp[2], 0 )
    print "timing results:"
    for e in epoch_timings: print "    epoch %s time: %s" % e

def read_vlaser( vlaser_string ):
    vlaser = ones(1)        
    try:
        vlaser = vlaser * float( vlaser_string )
    except:
        vlaser = FBF.read( vlaser_string, 1, -1 )
        vlaser = reshape( vlaser, ( vlaser.shape[0], ) )
    return vlaser    

### Command-line interface
if __name__=='__main__':
    def usage():
        print "USAGE: resampleWrapper [options] in_f vlaser"
        print "       options are:"
        print "       -f fbf_dir: is the data directory (default .)"
        print "       -s: single precision mode (default is double)"
        print "       -o outmod: modify outputname by substituting outmod_... for out_..."
        print "       -u outputname: use instead of default convention (supercedes -o)"
        print "       -r resamplerExec: use specified executable for resampler stage"
        print "       -z zero_threshold: use zero-thresholding optimization parameter (default 0)"
        print "       -i: vlaser is vlaser_in (value or file) (default is vlaser_ratio)"
        print "       -e epochs: how many times to repeat the test"
    fp_ref = None
    vlaser_str = None
    vlaser_in = None
    precision = DOUBLE_PRECISION
    fbf_dir = '.'
    f_in = None
    outname = 'out'
    fulloutname = None
    executable = RESAMPLING_EXECUTABLE
    zero_threshold = 0
    epochs = 1
    
    from getopt import getopt # get command line arguments
    try:
        opts, args = getopt( sys.argv[1:], 'f:so:u:r:z:ie:' )
        assert len( args ) == 2
        f_in, vlaser_str = args

        for o, a in opts:
            if o == '-f':
                fbf_dir = a
            elif o == '-s':
                precision = SINGLE_PRECISION
                executable = RESAMPLING_EXECUTABLE_SINGLE_PRECISION_VERSION
            elif o == '-o':
                outname = a
            elif o == '-u':
                fulloutname = a
            elif o == '-r':
                executable = a
            elif o == '-z':
                zero_threshold = int( a )
            elif o == '-i':
                vlaser_in = 1
            elif o == '-e':
                epochs = int( a )
            else:
                assert 0 # no other options accepted
            
    except: 
        usage()
        sys.exit( 1 )

    vlaser_val = read_vlaser( vlaser_str )
        
    fbf_in = FBF.FBF( os.path.join( fbf_dir, f_in ) ) 
    fbf_out_name = get_out_name( fbf_in, vlaser_str, 
                                 out_label = outname, 
                                 algorithm = 'ccnote', 
                                 precision = precision, 
                                 zero_threshold_param = zero_threshold )
    if fulloutname:
      fbf_out_name = fulloutname
    detector = 'LW'
    if fbf_in.grouping[0] == 2049:
        detector = 'SMW'
    if DEBUG:
        print fbf_out_name
        sys.exit( 0 )

    fbf_out = FBF.FBF( fbf_out_name )

    prep_fbf_input( fbf_in )
    prep_fbf_output( fbf_out, fbf_in.length() )

    if vlaser_in:
        gifts_main( fbf_in, fbf_out, detector, 
                    vlaser_in = vlaser_val, 
                    resampler = executable, 
                    zero_threshold_param = zero_threshold, 
                    num_epochs = epochs,
                    nocache = NOCACHE )
    else: # vlaser_ratio mode
        gifts_main( fbf_in, fbf_out, detector, 
                    vlaser_ratio = vlaser_val, 
                    resampler = executable, 
                    zero_threshold_param = zero_threshold, 
                    num_epochs = epochs, 
                    nocache = NOCACHE )
        
    fbf_in.close()
    fbf_out.close()
    
    print "done."
    sys.exit( 0 )
    
    
