#!/usr/bin/env python

""" Run a cube test:
Needs:
  both truth (on axis) and off axis rolled off FBF files
  a file with laser wavenumber ratios
  binary versions of both Sinc Resampler and FFT resampler
  Sufficient information about the input files to set the appropriate options on the resamplers
Does:
  prepares a directory for the test, with appropriate files softlinked in (using call to prepareResampleTestFBF)
  runs both sinc and FFT resamplers
  runs a diagnostic program 

"""

GIFTS_VLASER_NOMINAL = 1e7 / 1064

EXECTAB = {
'SINC_LW_DOUBLE': '/home/maciek/experiments/2004_06_24_resampler/src/c++/sinc/ResamplingStage',
'DFFT_LW_DOUBLE' : '/home/maciek/experiments/2004_06_24_resampler/src/c++/dfft/FFTResamplingStage',
#FIXME - the above should be generalized. In the meantime, change to suit your install.
#FIXME - create executlables for other combinations of SMW, SINGLE (precision):
'SINC_SMW_DOUBLE' : '',
'DFFT_SMW_DOUBLE' : '',

'SINC_LW_SINGLE' : '',
'DFFT_LW_SINGLE' : '',

'SINC_SMW_SINGLE' : '',
'DFFT_SINGLE' : '',
}
LENTAB = {'SMW': 2049, 'LW': 1025}
FBFTYPETAB = {'SINGLE': 'real4', 'DOUBLE': 'real8'}

from math import sqrt
from Numeric import *
from os import chdir
from sys import stderr, argv, exit
from FBF import FBF # SSEC Flat Binary Format file API

from prepareResampleTestFBF import genTestDir
from resampleWrapper2       import gifts_main, read_vlaser, prep_fbf_input, prep_fbf_output

def runErrorMetrics( test_name, truth_fbf, test_fbf, wn_scale ):
  DIFFMIN, DIFFMAX = 0, -1

  # what files we're generating
  
  fbf_out = {
    'delta': FBF( stemname='error'+test_name, grouping = truth_fbf.grouping, typename = 'real8' ),
    'maxdelta':  FBF( stemname='maxError'+test_name, grouping = [], typename = 'real8' ),
    'rmsdelta': FBF( stemname='rmsError'+test_name, grouping = [], typename = 'real8' ),
  }
  for d in fbf_out:
    fbf_out[d].create()

  # read the input files
  truth = truth_fbf.read( 1, -1 )
  test = test_fbf.read( 1, -1 )

  # introducing the output variables that'll get written to these files
  rawd = zeros( shape( truth ), 'd' )
  maxd, rmsd = zeros( len( truth ), 'd'), zeros( len( truth ), 'd' )
  
  # and run the comparisons
  for ridx in range( len( truth ) ):
    rawdtmp = truth[ridx] - test[ridx]
    # print ( "truth: %s test: %s out: %s" % (shape(truth), shape(test), shape(rawd)) ) #DEBUG
    rawd[ridx] = rawdtmp
    limd = rawd[ridx][154:938] #FIXME - this range is only valid for LW wavenumber assignment
    maxd[ridx] = max( limd )
    #Deficiency in Jython's Jnumeric: there is no "average" - thus the below
    rmsd[ridx] = sqrt( sum( limd * limd ) / 784 ) #FIXME - again, 784 is valid for LW only

  fbf_out['delta'].block_write( 1, rawd )
  fbf_out['maxdelta'].block_write( 1, maxd )
  fbf_out['rmsdelta'].block_write( 1, rmsd )

  for d in fbf_out:
    fbf_out[d].close()
  

def runFullTest( cubedir, precision ):
  # separate cubedir into cubeprefix and detectorname
  sep = cubedir.rfind( '_' )
  cube, detector = cubedir[:sep], cubedir[sep+1:]
  if detector not in ('LW', 'SMW'):
      print "detector must be LW or SMW"
      return

  # invoke testdir generation
  print >>stderr, 'TEST: generating dir'
  genTestDir( cube, detector ) 
  chdir( cubedir )

  # open and prepare the FBF files
  fbfin =  FBF( 'spectrumOffAxisRolled.%s.%s' % (FBFTYPETAB[precision], LENTAB[detector]) )
  prep_fbf_input( fbfin )
  fbf_truth = FBF( 'spectrumTruthRolled.%s.%s' % (FBFTYPETAB[precision], LENTAB[detector]) )
  rat = read_vlaser( 'wnRatio.real8' )

  #TESTS = ('DFFT',)  #uncomment this line for fast, if limited test (no sinc resampler testing)
  TESTS = ('DFFT','SINC') # uncomment this line for slower, comprehensive test of both resamplers
  
  for test in TESTS:
    print >>stderr, 'TEST: running test for ' + test
    fbfin.fp().seek( 0 )
    fbfout = FBF( stemname = 'spectrumCorrected%s' % test, typename = FBFTYPETAB[precision], grouping = [LENTAB[detector],] )
    prep_fbf_output( fbfout, fbf_truth.length() )
    resampler_exec = "%s_%s_%s" % (test, detector, precision)
    gifts_main( fbfin, fbfout, detector,
                    vlaser_out = GIFTS_VLASER_NOMINAL,
                    vlaser_ratio = rat,
                    resampler = EXECTAB[resampler_exec], 
                    nocache = {'DFFT':0, 'SINC':1}[test] )
  
    runErrorMetrics( test, fbf_truth, fbfout, None )
    fbfout.close()
  fbfin.close()
  fbf_truth.close()

if __name__ == '__main__':
  if len( argv ) != 3 or not (argv[2] == 'SINGLE' or argv[2] == 'DOUBLE'):
    print 'USAGE: %s dirname (SINGLE|DOUBLE)' % argv[0]
    exit( 1 )
  runFullTest( argv[1], argv[2] )

