/** 
*  \file BindCdf.hxx
*  \brief Tool templates for moving NetCDF data to and from memory
*    
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: BindCdf.hxx,v 1.24.2.1 2005/12/15 19:38:14 graemem Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*  \endverbatim
*/


/** Interface specification

Idea: model one or more netcdf variables as an iterable data structure sequence.

Note: The NcFile and NcVars which are being bound must exist for at least as long as all 
binders which use them!  

*/

#ifndef HXX_BINDCDF
#define HXX_BINDCDF

#include <unistd.h>

#include <algorithm>
#include <vector>

#include <boost/shared_ptr.hpp>

#include "netcdfcpp.h"



// This is not CDF-specific. 

class FileToMemBinder
{
public:
    virtual void operator()( int, int ) = 0;     // record number to write to, record to read from; <0 for none
    virtual ~FileToMemBinder() { }
    class ioError // exception raised if the transfer fails
    {
    };
    class seekError // could not go to a given record
    {
    };
};    

typedef boost::shared_ptr<FileToMemBinder> FileToMemBinder_P;

/** 
* We want a group of Fields to move easily from record to record in unison. 
* 
* FIXME: either make this behave FULLY as an iterator, or move that behavior to a different class!
*
*/
class FileToMemBinderGroup
{
    typedef std::pair<int,int> recordchange;
    typedef std::vector<FileToMemBinder_P> binder_v;
    binder_v binders;
    recordchange recs;
        
    void updateBinder( binder_v::value_type b )
    {
        b->operator()(recs.first,recs.second);
    }
    
    void update_()
    {
        //std::for_each(binders.begin(), binders.end(), std::mem_fun(&FileToMemBinderGroup::updateBinder));
        for( binder_v::iterator i=binders.begin(); i!=binders.end(); i++ )
            updateBinder( *i );
    }
    
    public:

/// initialize with a null-terminated list of FileToMemBinders to be controlled. 
// ownership of the binders is not transferred.     
    explicit FileToMemBinderGroup(  ): recs(-1,-1)
    {
    }

    // copy constructor, allowing us to put these things in STL containers
    FileToMemBinderGroup( const FileToMemBinderGroup &orig ): binders(orig.binders), recs(orig.recs)
    {
    }

    
    FileToMemBinderGroup &push_back( FileToMemBinder_P pbin )
    {
        binders.push_back(pbin);
        return *this;
    }
    
    ~FileToMemBinderGroup()
    {
        recs = recordchange(recs.second,-1);
        // flush anything going out
        update_();
    }
    
    // move to a given record at absolute or relative 
    // return new record number
    // do any reads and writes needed to update the bound memory
    // throws seekError, ioError
    // FUTURE: consider return 0==success, <0 on failure rather than throwing exceptions
    int update( int offset, int whence )
    {
        recs.first = recs.second;
        switch(whence)
        {
            case SEEK_SET: recs.second = offset; break;
            case SEEK_CUR: recs.second += offset; break;
            default: throw FileToMemBinder::seekError(); 
        }
        update_();
        return recs.second;
    }
};    


#if 0

// FIXME: place this code into a fully iterator-like wrapper for the bound memory!

    FileToMemBinderGroup& operator+=( int nrecords ) 
    { 
        update(
        recs = recordchange(recs.second,recs.second+nrecords);
        update_();
        return *this; 
    }
    FileToMemBinderGroup& operator-=(int nrecords) { return operator+=(-nrecords); }
    
    FileToMemBinderGroup& operator++() { return operator+=(1); }
    FileToMemBinderGroup& operator++(int) { return operator+=(1); }
    FileToMemBinderGroup& operator--() { return operator-=(-1);  }
    FileToMemBinderGroup& operator--(int) { return operator-=(-1); }    
#endif





///
/// CDF specific stuff
///

// build a compile-time type lookup system!
template< typename T >
struct nc_traits
{ 
    typedef T value_type;
    typedef T *pointer;
    typedef T &reference;
    static const NcType nctype = ncNoType;
};
  
#define CDFTRAITS(cct,nct) struct nc_traits<cct>  \
{ \
    typedef cct value_type; \
    typedef cct *pointer; \
    typedef cct &reference; \
    static const NcType nctype = nct; \
}; 

CDFTRAITS(double,ncDouble)
CDFTRAITS(float,ncFloat)
CDFTRAITS(char,ncChar)
CDFTRAITS(short,ncShort)
CDFTRAITS(int,ncInt)
CDFTRAITS(unsigned char,ncByte)


typedef NcVar *NcVar_p; // NOT! boost::shared_ptr<NcVar> NcVar_p; since NcFile retains ownership!


template< typename A, typename B > // already exists in std::?? 
inline A unary_instance( const B &b ) { return A(b); }



// A utility class to aid the transfer of data between memory and CDF files.
// Specifically, compile-time typing of C++ versus run-time typing of CDF files.
template<typename T>
class nctool
{
protected:
    NcVar_p var;
    const unsigned ndims;
    const NcType vartype;
    void *buffer;
    int buffer_count;
    
    // find the total number of elements to copy
    // assumes that counts[] is proper length - i.e. one entry for each cdf dimension for the variable
    inline long prod(const long *counts) 
    {
        long n=1;
        for( unsigned i=0; i<ndims; i++ )
        {
            n *= *counts;
            ++counts;
        }
        return n;
    }
    
    template<typename TypeInFile>
    NcBool get_with_copy(T *data, const long *counts)
    {
//      FIXME: warn about mismatched data precision at constructor time to 
//      avoid copious warnings during runtime
//        if (sizeof(T) < sizeof(TypeInFile)) 
//            std::cerr << "WARNING: copying from CDF data with more bits\n";        
        long nelem = prod(counts);
        if (buffer_count < nelem)
        {
            buffer = realloc(buffer,sizeof(TypeInFile)*nelem);
            assert(buffer!=NULL);
        }
        TypeInFile *sbuf = (TypeInFile *)buffer;
        NcBool ret = var->get( sbuf, counts );
        if (ret)
        {
            std::transform( sbuf, sbuf+nelem, data, unary_instance<T,TypeInFile> );
            //std::copy( sbuf, sbuf+nelem, data );
        }
        return ret;
    }

    template<typename TypeInFile>
    NcBool put_with_copy(const T *data, const long *counts)
    {
//      FIXME: warn about mismatched data precision at constructor time to 
//      avoid copious warnings during runtime
//        if (sizeof(T) > sizeof(TypeInFile)) 
//            std::cerr << "WARNING: copying to CDF data with fewer bits\n";        
        long nelem = prod(counts);
        if (buffer_count < nelem)
        {
            buffer = realloc(buffer,sizeof(TypeInFile)*nelem);
            assert(buffer!=NULL);
        }
        TypeInFile *sbuf = (TypeInFile *)buffer;
        std::transform( data, data+nelem, sbuf, unary_instance<TypeInFile,T> );
        NcBool ret = var->put( sbuf, counts );
        return ret;
    }

public:
    nctool(NcVar_p ncv_): 
        var(ncv_), ndims(var->num_dims()), vartype(var->type()), buffer(NULL), buffer_count(0)
    { 
        buffer = malloc(1);
    }
    
    ~nctool()
    {
        free(buffer);
    }
    
    // assumption: counts points to an array at least var->num_dims() long
    NcBool get( T *data, const long *counts )
    {
        if (vartype != nc_traits<T>::nctype)
        {
            switch(vartype)
            {
                case ncDouble: return get_with_copy<double>( data, counts );
                case ncFloat: return get_with_copy<float>( data, counts );
                case ncByte: return get_with_copy<unsigned char>( data, counts );
                case ncInt: return get_with_copy<int>( data, counts );
                case ncChar: return get_with_copy<char>( data, counts );
                case ncShort: return get_with_copy<short>( data, counts );
                case ncNoType: abort( );
            }
        }
        return var->get(data,counts);
    }

    // assumption: counts points to an array at least var->num_dims() long
    NcBool put( const T *data, const long *counts )
    {
        if (vartype != nc_traits<T>::nctype)
        {
            switch(vartype)
            {
                case ncDouble: return put_with_copy<double>( data, counts );
                case ncFloat: return put_with_copy<float>( data, counts );
                case ncByte: return put_with_copy<unsigned char>( data, counts );
                case ncInt: return put_with_copy<int>( data, counts );
                case ncChar: return put_with_copy<char>( data, counts );
                case ncShort: return put_with_copy<short>( data, counts );
                case ncNoType: abort( );
            }
        }
        return var->put(data,counts);
    }  
};




// Similarly, a tool to move data to and from a pair of variables representing a complex dataset.

template< typename V >
class nccplxtool
{
protected:
    NcVar *var_real, *var_imag;
    const unsigned ndims;
    const NcType vartype;
    void *real, *imag;
    size_t buffer_size;
    
    template< typename SrcType, typename DstType >
    struct getreal
    {
        inline DstType operator()( const SrcType &src ) const
        { return DstType(src.real()); }
    };

    template< typename SrcType, typename DstType >
    struct getimag
    {
        inline DstType operator()( const SrcType &src ) const
        { return DstType(src.imag()); }
    };
    
    template< typename SrcType, typename DstType >
    struct merge
    {
        inline DstType operator()( const SrcType &re, const SrcType &im ) const
        { return DstType( re,im ); }
    };

    template< typename BufferType >
    inline void reserve( size_t n_elements )
    {
        size_t newsize = sizeof(BufferType)*n_elements;
        if (buffer_size < newsize)
        {
            real = realloc( real, newsize );
            imag = realloc( imag, newsize );
            buffer_size = newsize;
        }
    }

    template< typename TypeInFile >
    NcBool get_with_copy( V &data, const long *counts )
    {
        reserve<TypeInFile>( data.size() );
        if (!var_real->get( (TypeInFile *)real, counts )) return NcBool(0);
        if (var_imag!=NULL) 
        {
            if (!var_imag->get( (TypeInFile *)imag, counts )) return NcBool(0);
        }
        else
            bzero(imag,buffer_size);
        std::transform( (const TypeInFile *)real, ((const TypeInFile *)real)+data.size(), (const TypeInFile *)imag, 
                        data.begin(), merge<TypeInFile,typename V::value_type>() );
        return NcBool(1);
    }

    template< typename TypeInFile >
    NcBool put_with_copy( const V &data, const long *counts )
    {
        reserve<TypeInFile>( data.size() );
        // extract real and imaginary parts to temporary buffer
        std::transform( data.begin(), data.end(), (TypeInFile *)real, 
                            getreal<typename V::value_type,TypeInFile>() );
        if (!var_real->put( (TypeInFile *)real, counts )) return NcBool(0);
        if (var_imag)
        {
            std::transform( data.begin(), data.end(), (TypeInFile *)imag, 
                            getimag<typename V::value_type,TypeInFile>() );
            if (!var_imag->put( (TypeInFile *)imag, counts )) return NcBool(0);
        }
        return NcBool(1);
    }
        
public:
    nccplxtool(NcVar_p ncvr_, NcVar_p ncvi_): 
        var_real(ncvr_), var_imag(ncvi_), ndims(var_real->num_dims()), vartype(var_real->type()), buffer_size(0)
    { 
        real = malloc(1);
        imag = malloc(1);
        assert( var_imag==NULL || vartype == var_imag->type() );
    }
    
    ~nccplxtool()
    {
        free(real);
        free(imag);
    }

    // assumption: counts points to an array at least var->num_dims() long
    NcBool get( V &data, const long *counts )
    {
        switch(vartype)
        {
            case ncDouble: return get_with_copy<double>( data, counts );
            case ncFloat: return get_with_copy<float>( data, counts );
            case ncByte: return get_with_copy<unsigned char>( data, counts );
            case ncInt: return get_with_copy<int>( data, counts );
            case ncChar: return get_with_copy<char>( data, counts );
            case ncShort: return get_with_copy<short>( data, counts );
            case ncNoType:
            default:
                     abort( ); 
        }
        return NcBool(0);
    }

    // assumption: counts points to an array at least var->num_dims() long
    NcBool put( const V &data, const long *counts )
    {
        switch(vartype)
        {
            case ncDouble: return put_with_copy<double>( data, counts );
            case ncFloat: return put_with_copy<float>( data, counts );
            case ncByte: return put_with_copy<unsigned char>( data, counts );
            case ncInt: return put_with_copy<int>( data, counts );
            case ncChar: return put_with_copy<char>( data, counts );
            case ncShort: return put_with_copy<short>( data, counts );
            case ncNoType:
            default:
                     abort( ); 
        }
        return NcBool(0);
    }        
};



struct CdfRecordMapping 
{
  virtual NcBool operator()( NcVar_p&, unsigned ) = 0;  
};


// determines how dimensions of a variable are traversed
// example: we may want to read only one pixel of data at a time; 
// this implies traversing 0..N-1 records with 0..127 by 0..127 parts
// alternate record mapping tool
struct SimpleCdfRecordMapping : public CdfRecordMapping
{
  //public:

  SimpleCdfRecordMapping(): CdfRecordMapping() {}
  
  NcBool operator()( NcVar_p& var, unsigned n )
  { 
    return var->set_cur(n); 
  }    
};

template< typename TypeInMem > 
class CdfFieldReader: public FileToMemBinder
{
  TypeInMem *field;
  NcVar_p var;
  CdfRecordMapping *seek;
  nctool<TypeInMem> getput;  
  
public:
  CdfFieldReader(TypeInMem *field_, NcVar_p var_, CdfRecordMapping *seek_): 
      field(field_), var(var_), seek(seek_), getput(var) 
      {}

  virtual void operator()( int writetorec, int readfromrec )
  {
    NcError ncerr( NcError::verbose_nonfatal );
    if (readfromrec < 0) return;
    if (!seek->operator()( var, readfromrec )) throw seekError(); 
    long count[] = { 1, 0 };
    if (!getput.get( field, &count[0] )) throw ioError();
  }
};


template< typename TypeInMem >
class CdfFieldWriter: public FileToMemBinder
{
  const TypeInMem *field;
  NcVar_p var;
  CdfRecordMapping *seek;
  nctool<TypeInMem> getput;
  
public:
  CdfFieldWriter(const TypeInMem *field_, NcVar_p var_, CdfRecordMapping *seek_): 
      field(field_), var(var_), seek(seek_), getput(var) 
      { }
  
  virtual void operator()( int writetorec, int readfromrec )
  {
        NcError ncerr( NcError::verbose_nonfatal );
        if (writetorec < 0) return;
        if (!seek->operator()( var, writetorec )) throw seekError(); 
        long count[] = { 1, 0 };
        if (!getput.put( field, &count[0] )) throw ioError();
  }
};


template< typename TypeInMem >
class CdfVectorReader: public FileToMemBinder
{
    TypeInMem *field;
    NcVar_p var;
    CdfRecordMapping *seek;
    nctool<typename TypeInMem::value_type> getput;
    std::vector<long> widths;
    
public:
    template<typename Iterlike>
    CdfVectorReader(TypeInMem *field_, NcVar_p var_, 
                    Iterlike beginwidths, Iterlike endwidths, 
                    CdfRecordMapping *seek_): 
        field(field_), 
        var(var_),
	seek(seek_),
        getput(var),
        widths(beginwidths,endwidths)
    { 
      assert( widths.size() == (size_t)var->num_dims() );
    }
    
    virtual void operator()( int writetorec, int readfromrec )
    {
        NcError ncerr( NcError::verbose_nonfatal );
        if (readfromrec < 0) return;
        if (!seek->operator()( var, readfromrec )) throw seekError(); 
        if (!getput.get( &(*field)[0], &widths[0] )) throw ioError();
    }
};


template< typename TypeInMem >
class CdfVectorWriter: public FileToMemBinder
{
    const TypeInMem *field;
    NcVar_p var;
    CdfRecordMapping *seek;
    nctool<typename TypeInMem::value_type> getput;
    std::vector<long> widths;
    
public:
    template<typename Iterlike>
    CdfVectorWriter(const TypeInMem *field_, NcVar_p var_, 
                    Iterlike beginwidths, Iterlike endwidths, 
                    CdfRecordMapping *seek_): 
        field(field_), 
        var(var_),
	seek(seek_),
        getput(var),
        widths(beginwidths,endwidths)
    { 
        assert( widths.size() == (size_t)var->num_dims() );
    }
    
    virtual void operator()( int writetorec, int readfromrec )
    {
        NcError ncerr( NcError::verbose_nonfatal );
        if (writetorec < 0) return;
        if (!seek->operator()( var, writetorec )) throw seekError(); 
        if (!getput.put( &(*field)[0], &widths[0] )) throw ioError();
    }
};



template< typename TypeInMem >
class CdfCplxVectorReader: public FileToMemBinder
{
    TypeInMem *field;
    NcVar_p var_real, var_imag;
    CdfRecordMapping *seek;
    nccplxtool<TypeInMem> getput;
    std::vector<long> widths;
    
public:
    template<typename Iterlike>
    CdfCplxVectorReader(TypeInMem *field_, NcVar_p var_real_, NcVar_p var_imag_, 
                    Iterlike beginwidths, Iterlike endwidths, 
                    CdfRecordMapping *seek_): 
        field(field_), 
        var_real(var_real_),
        var_imag(var_imag_),
        seek(seek_),
        getput(var_real, var_imag),
        widths(beginwidths,endwidths)
    { 
        assert( widths.size() == (size_t)var_real->num_dims() );
        assert( widths.size() == (size_t)var_imag->num_dims() );
    }
    
    virtual void operator()( int writetorec, int readfromrec )
    {
        NcError ncerr( NcError::verbose_nonfatal );
        if (readfromrec < 0) return;
        if (!seek->operator()( var_real, readfromrec )) throw seekError(); 
        if (!seek->operator()( var_imag, readfromrec )) throw seekError(); 
        if (!getput.get( *field, &widths[0] )) throw ioError();
    }
};


template< typename TypeInMem >
class CdfCplxVectorWriter: public FileToMemBinder
{
    const TypeInMem *field;
    NcVar_p var_real, var_imag;
    CdfRecordMapping *seek;
    nccplxtool<TypeInMem> getput;
    std::vector<long> widths;
    
public:
    template<typename Iterlike>
    CdfCplxVectorWriter(const TypeInMem *field_, NcVar_p var_real_, NcVar_p var_imag_, 
                    Iterlike beginwidths, Iterlike endwidths, 
                    CdfRecordMapping *seek_): 
        field(field_), 
        var_real(var_real_),
        var_imag(var_imag_),
        seek(seek_),
        getput(var_real, var_imag),
        widths(beginwidths,endwidths)
    { 
        assert( widths.size() == (size_t)var_real->num_dims() );
        assert( widths.size() == (size_t)var_imag->num_dims() );
    }
    
    virtual void operator()( int writetorec, int readfromrec )
    {
        NcError ncerr( NcError::verbose_nonfatal );
        if (writetorec < 0) return;
        if (!seek->operator()( var_real, writetorec )) throw seekError(); 
        if (!seek->operator()( var_imag, writetorec )) throw seekError(); 
        if (!getput.put( *field, &widths[0] )) throw ioError();
    }
};



// map 16384 pixels as sequential records, assuming only two dimensions (I and J)
struct GiftsPixelRecordMapping: public CdfRecordMapping
{
    NcBool operator()( NcVar_p& var, unsigned n )
    { 
        if (n>=16384) return NcBool(0);
        long where[] = { n/128, n%128, 0 };
        return var->set_cur(&where[0]); 
    }    
};


// 
// Template functions to act as factories with implicit type binding
// Mostly syntactical sugar to avoid having to restate the type of the in-memory target: 
// 
//  shifter.push_back( FileToMemBinder_P( new CdfFieldReader<double>( &abbApexTemp, NcVar_p( fp->get_var("abbApexTemp")))));
//
//  becomes
//
// shifter.push_back( newCdfFieldReader( &abbApexTemp, NcVar_p(fp->get_var("abbApexTemp")) ));
// 

template< typename InMemory >
FileToMemBinder_P newCdfFieldReader( InMemory *mem, NcVar_p var, CdfRecordMapping *seek )
{ return FileToMemBinder_P(new CdfFieldReader<InMemory>( mem, var, seek )); }

template< typename InMemory >
FileToMemBinder_P newCdfFieldWriter( const InMemory *mem, NcVar_p var, CdfRecordMapping *seek )
{ return FileToMemBinder_P(new CdfFieldWriter<InMemory>( mem, var, seek )); }

template< typename InMemory, typename Iterlike >
FileToMemBinder_P newCdfVectorReader( InMemory *mem, NcVar_p var, Iterlike wbegin, Iterlike wend, CdfRecordMapping *seek )
{ return FileToMemBinder_P(new CdfVectorReader<InMemory>( mem, var, wbegin, wend, seek )); }

template< typename InMemory, typename Iterlike >
FileToMemBinder_P newCdfVectorWriter( const InMemory *mem, NcVar_p var, Iterlike wbegin, Iterlike wend, CdfRecordMapping *seek )
{ return FileToMemBinder_P(new CdfVectorWriter<InMemory>( mem, var, wbegin, wend, seek )); }

template< typename InMemory, typename Iterlike >
FileToMemBinder_P newCdfCplxVectorReader( InMemory *mem, NcVar_p var_re, NcVar_p var_im, Iterlike wbegin, Iterlike wend, CdfRecordMapping *seek )
{ return FileToMemBinder_P(new CdfCplxVectorReader<InMemory>( mem, var_re, var_im, wbegin, wend, seek )); }

template< typename InMemory, typename Iterlike >
FileToMemBinder_P newCdfCplxVectorWriter( const InMemory *mem, NcVar_p var_re, NcVar_p var_im, Iterlike wbegin, Iterlike wend, CdfRecordMapping *seek )
{ return FileToMemBinder_P(new CdfCplxVectorWriter<InMemory>( mem, var_re, var_im, wbegin, wend, seek )); }


#endif
