/** 
*  \file BindCdfTest.hxx
*  \brief Unit test for CDF binding utility classes
*    
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: BindCdfTest.cc,v 1.10.2.1 2005/12/15 21:03:29 rayg Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* FUTURE: consider parallelizing file I/O. Possibly group readers and 
* writers separately within a Binder Group and specify no sharing of 
* resources among different binders to prevent race conditions. Binders
* can themselves be multi-threaded if desired.
*
*
*  \endverbatim
*/


#include <complex>

#include "BindCdf.hxx"


struct TestWorkspace
{
    double abbApexTemp, hbbApexTemp, Time;
    double timeHHMMSS;
    typedef std::vector<double> double_v;
    double_v xphase, yphase;
    SimpleCdfRecordMapping *mapping;

    NcFile *fp;    
    FileToMemBinderGroup shifter;
    
    TestWorkspace( const char *fn ): xphase(34), yphase(34)
    {
        fp = new NcFile( fn );
        mapping = new SimpleCdfRecordMapping(); 
        
        shifter.push_back( newCdfFieldReader( &Time, NcVar_p(fp->get_var("Time")), mapping ));
        shifter.push_back( newCdfFieldReader( &timeHHMMSS, NcVar_p(fp->get_var("timeHHMMSS")), mapping ));
        shifter.push_back( newCdfFieldReader( &abbApexTemp, NcVar_p(fp->get_var("ABBapexTemp")), mapping ));
        shifter.push_back( newCdfFieldReader( &hbbApexTemp, NcVar_p(fp->get_var("HBBapexTemp")), mapping ));
        
        long width[] = { 1 /* record */, xphase.size() /* points */ };
        shifter.push_back( newCdfVectorReader( &xphase, NcVar_p(fp->get_var("errorXphase")), width, width+2, mapping ) );        
        shifter.push_back( newCdfVectorReader( &yphase, NcVar_p(fp->get_var("errorYphase")), width, width+2, mapping ) );        
    }    
};


void test1( const char *filename )
{
    TestWorkspace tw( filename );
    tw.shifter.update( 0, SEEK_SET );
    int rec = 0;
    for(;;rec++) 
    {
        if (rec<5)
        {
            std::cout << tw.Time << '=' << tw.timeHHMMSS << ':' << tw.abbApexTemp 
                        << ' ' << tw.hbbApexTemp << std::endl << '\t';
            for( int ii=0; ii<10; ++ii ) std::cout << tw.xphase[ii] << ',' << tw.yphase[ii] << ' ';
            std::cout << std::endl;
        }
        try
        { 
            tw.shifter.update( +1, SEEK_CUR );
            std::cout << rec << "   \n";
        }
        catch( FileToMemBinder::ioError io )
        {
            std::cout << "\nioerror\n";
            break;
        }
        catch( FileToMemBinder::seekError io )
        {
            std::cout << "\nseekerror\n";
            break;
        }
    }
}

void test2( )
{
    std::vector<float> data( 2048 );
    
    unlink("bindcdf_test2.cdf");
    NcFile fp( "bindcdf_test2.cdf", NcFile::New );
    assert(fp.is_valid());
    NcDim *dim = fp.add_dim("index",data.size());
    assert(dim);
    NcVar *var = fp.add_var("signalReal",ncFloat,dim);
    assert(var);
    
    SimpleCdfRecordMapping mapping;
    FileToMemBinderGroup shifter;
    long width[] = { data.size() };
    shifter.push_back( newCdfVectorWriter( &data, var, width, width+1, &mapping ) );

    shifter.update(0,SEEK_SET);
    for( unsigned i=0; i<data.size(); i++ )
        data[i] = float(i);
    
    // write, flush, close on dtors
}

void test3( )
{
    std::vector<float> data( 2048 );
    
    unlink("bindcdf_test3.cdf");
    NcFile fp( "bindcdf_test3.cdf", NcFile::New );
    assert(fp.is_valid());
    NcDim *dim0= fp.add_dim("I",8);
    assert(dim0);
    NcDim *dim1= fp.add_dim("J",8);
    assert(dim1);
    NcDim *dim2= fp.add_dim("wnum",data.size());
    assert(dim2);
    NcVar *var = fp.add_var("signalReal",ncFloat,dim0,dim1,dim2);
    assert(var);
    
    GiftsPixelRecordMapping mapping;
    FileToMemBinderGroup shifter;
    long width[] = { 1, 1, data.size() };
    shifter.push_back( newCdfVectorWriter( &data, var, width, width+3, &mapping ) );

    shifter.update(0,SEEK_SET);
    for( unsigned i=0; i<data.size(); i++ )
        data[i] = float(i);
    
    // write, flush, close on dtors
}

void test4( )
{
    std::vector<std::complex<float> > data( 2048 );
    
    unlink("bindcdf_test4.cdf");
    NcFile fp( "bindcdf_test4.cdf", NcFile::New );
    assert(fp.is_valid());
    NcDim *dim0= fp.add_dim("I",8);
    assert(dim0);
    NcDim *dim1= fp.add_dim("J",8);
    assert(dim1);
    NcDim *dim2= fp.add_dim("wnum",data.size());
    assert(dim2);
    NcVar *varre = fp.add_var("signalReal",ncFloat,dim0,dim1,dim2);
    assert(varre);
    NcVar *varim = fp.add_var("signalImag",ncFloat,dim0,dim1,dim2);
    assert(varim);
    
    GiftsPixelRecordMapping mapping;
    FileToMemBinderGroup shifter;
    long width[] = { 1, 1, data.size() };
    shifter.push_back( newCdfCplxVectorWriter( &data, varre, varim, width, width+3, &mapping ) );

    shifter.update(0,SEEK_SET);
    for( unsigned i=0; i<data.size(); i++ )
        data[i] = std::complex<float>(float(i),-float(i));
    
    // write, flush, close on dtors
}

void test4a( )
{
    std::vector<std::complex<double> > data( 2048 );
    
    NcFile fp( "bindcdf_test4.cdf" );
    assert(fp.is_valid());
    NcVar *varre = fp.get_var("signalReal");
    assert(varre);
    NcVar *varim = fp.get_var("signalImag");
    assert(varim);
    
    GiftsPixelRecordMapping mapping;
    FileToMemBinderGroup shifter;
    long width[] = { 1, 1, data.size() };
    shifter.push_back( newCdfCplxVectorReader( &data, varre, varim, width, width+3, &mapping ) );

    shifter.update(0,SEEK_SET);
    for( unsigned i=0; i<8; /* i<data.size();*/ i++ )
        std::cout << data[i] << '\t';
    std::cout << std::endl;
    for( unsigned i=1024; i<1032; /* i<data.size();*/ i++ )
        std::cout << data[i] << '\t';
    std::cout << std::endl;
    for( unsigned i=2040; i<2048; /* i<data.size();*/ i++ )
        std::cout << data[i] << '\t';
    std::cout << std::endl;
    
    // write, flush, close on dtors
}




int main( int argc, char *argv[] ) 
{ 
    test4();
    test4a();
    test3();
    test2();
    if(argc==2)
        test1(argv[1]);
    return 0;
}


