
/** 
*  \file PiecewiseLinearCalibrationContext.hxx
*  \brief Template for modeling blackbody temperatures as linear windows in time.
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: PiecewiseLinearCalibrationContext.hxx,v 1.9.2.5 2005/12/15 23:13:34 rayg Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*  \endverbatim
*/


#define DEBUG
#ifdef DEBUG
#define DBG(x) x
#else
#define DBG(x) 
#endif

#include <algorithm>
#include <iostream>
#include <vector>
#include <map> 
#include <complex>
#include <boost/shared_ptr.hpp>

namespace gips 
{

// FIXME: Promote this to outside PLCC.hxx!
class PolynomialCalibrationContext
{
protected:
    typedef std::vector< std::complex<double> > CoeffVector;

public:
    const unsigned SIZE,COEFFS;        
    double baseT;
    CoeffVector *A;
    
public:    
    void clear() 
    {
        baseT = 0.0;        
        for( unsigned c=0; c<COEFFS; c++ )
            std::fill( A[c].begin(), A[c].end(), 0 );
    }
    
    explicit PolynomialCalibrationContext( unsigned size, unsigned order = 3 ):
        SIZE(size),
        COEFFS(order+1)
    {
        A = new CoeffVector [ COEFFS ];        
        for( unsigned c=0; c<COEFFS; c++ )
            A[c].resize( SIZE );
    }

    PolynomialCalibrationContext( const PolynomialCalibrationContext &src ):
        SIZE(src.SIZE), COEFFS(src.COEFFS)
    {
        A = new CoeffVector [ COEFFS ];
        for( unsigned c=0; c<COEFFS; c++ )
            A[c] = src.A[c];
    }
    
    ~PolynomialCalibrationContext()
    {
        delete [] A;
    }
        
    template< typename Target >
    void operator()( Target &X, double T ) const
    {
        assert( X.size() >= A[0].size() );
        static const CoeffVector::value_type zero(0);
        
        T -= baseT;        
        for( unsigned w=0; w < SIZE; ++w )
        {
            typename Target::value_type Y = A[1][w] * T + A[0][w];
            for( unsigned c=2; c < COEFFS; ++c )
            {
                CoeffVector::value_type a( A[c][w] );
                if (a != zero) Y += a * pow(T,int(c));
            }
            X[w] = Y;
        }
    }

// information on how to cache these things quickly and painlessly
// unless you have a more involved way to deal with storing/serializing/deserializing

    typedef std::pair< char, char > k_scene; // H/A/Z, F/B
    typedef std::pair< k_scene, double > k_cache; // scene, start time 

    static k_cache key( char scenetype, char scandir, double time ) 
    {
        return k_cache( k_scene( scenetype, scandir ), time );
    }

    typedef std::map< k_cache, boost::shared_ptr<PolynomialCalibrationContext> > BasicCacheType;
    
    template<typename Iteratorlike>    
    inline static bool sanityCheckCacheEntry( char scenetype, char scandir, Iteratorlike x )
    {
      DBG( std::clog << "sanity check: " << x->first.first.first << x->first.first.second << 'v' << scenetype << scandir << " time: " << x->first.second << std::endl );
        // make sure that we're returning something that actually is ffor this scene and direction!
        return bool(x->first.first == k_scene(scenetype,scandir));
   }

};




template
< 
    typename InType, 
    typename CacheType 
>
class PiecewiseLinearCalibrationStudy
{
protected:
    typedef PolynomialCalibrationContext PCC;
    
public:
    
    struct settings_t
    {
        unsigned spectrumSize;
        
        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.spectrumSize = x.spectrumSize;
            return ret;
        }        
    };

    struct ancillary_t 
    {
        double time; // linear time representation
        char scenetype; // H/A/Z/E
        char scandir; // F/B

        template< typename Src > 
        static ancillary_t fromStruct( const Src &x )
        {
            ancillary_t ret;
            ret.time = x.time;
            ret.scenetype = x.scenetype;
            ret.scandir = x.scandir;
            return ret;
        }
    };
    
    struct Ports 
    {
        const InType *obs_in;
        const ancillary_t *anc_in;
        CacheType *cache_out;

        Ports(const InType *obs_in_, const ancillary_t *anc_in_, CacheType *cache_out_):
            obs_in(obs_in_),
            anc_in(anc_in_),
            cache_out(cache_out_)
        { }
    };
    
protected:

    const settings_t settings;
    Ports P;
    
    // all state for this capsule is in the cache!
    
    // start is the previous calibration frame, which has previous time, and previous spectrum in A[0]
    void calculate_entry( PCC &out, const PCC &start ) 
    {
        out.clear();
        
        // starting time is 0.0 - (anc_in->time - prevT)
        // ending time is 0.0, corresponding to anc_in->time
        // This makes it trivial to recover the spectrum at anc_in->time from the PCC
        
        double deltaT = P.anc_in->time - start.baseT;
        out.baseT = P.anc_in->time;
        const InType &oin( *P.obs_in );
        
        for( unsigned w=0; w < settings.spectrumSize; ++w )
        {
            out.A[0][w] = oin[w];
            out.A[1][w] = (oin[w] - start.A[0][w]) / deltaT;
        }
    }

    // create initial cache entry going back to T=0, zeroed A[1..n]
    void make_initial_entry( PCC &out )
    {
        out.clear();
        out.baseT = P.anc_in->time;
        const InType &oin( *P.obs_in );

        for( unsigned w=0; w < settings.spectrumSize; ++w )
            out.A[0][w] = oin[w];
        DBG( std::clog << "initializing for " << P.anc_in->scenetype << P.anc_in->scandir << std::endl );
    }
    
    template<typename CacheIterator, typename CacheKey >
    int update( CacheIterator iter, const CacheKey &kiki )
    {
        CacheType &cache_out( *P.cache_out );
        // if the iterator isn't the start of the whole cache,
        // and it's not a perfect match for the data, 
        // go to the previous entry
        // since lower_bound returns the iterator pointing to the first element with a key not less than k,
        // but what we want is the last key which is less than or equal to k!
        if ( (iter != cache_out.begin()) && (iter->first != kiki) )
            --iter;
    
        boost::shared_ptr<PCC> newcontext( new PCC(settings.spectrumSize,1) );
        if ( (iter == cache_out.end()) || 
                !PCC::sanityCheckCacheEntry(P.anc_in->scenetype,P.anc_in->scandir,iter) )
        {
            make_initial_entry( *newcontext );
            cache_out[ PCC::key(P.anc_in->scenetype, P.anc_in->scandir, 0) ] = newcontext;
        }
        else
        {
            // use the entry at *iter as the starting point for a new context
            // base time of that context is the starting time (and thus used in the key, so lower_bound works)
            calculate_entry( *newcontext, *(iter->second) );
            cache_out[ PCC::key(P.anc_in->scenetype, P.anc_in->scandir, iter->second->baseT) ] = newcontext;
        }
        return 0;
    }
    
    template< typename CacheKey > // is a little bit silly... okay more than a little bit
    int update( const CacheKey &kiki )
    {
        CacheType &cache_out( *P.cache_out );
        return update( cache_out.lower_bound( kiki ), kiki );
    }

public:

    PiecewiseLinearCalibrationStudy( const settings_t &settings_, const Ports &p_ ): settings(settings_), P(p_) { }

    int operator()( ) 
    {
        assert( settings.spectrumSize <= P.obs_in->size() );
        // recover most recent entry for this scene type from cache
        // create a new one based on it; if it doesn't exist, initialize for this scene type
        return update( PCC::key(P.anc_in->scenetype, P.anc_in->scandir, P.anc_in->time) );
        
        return 0;  // FIXME: return failure if we get a division by zero 
    }    
};


template
< 
    typename CacheType,
    typename OutType 
>
class ThreeBodyPolynomialCalibrationSynthesis
{
protected:
    typedef PolynomialCalibrationContext PCC;

public:

    struct settings_t
    {
        unsigned spectrumSize;
        
        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.spectrumSize = x.spectrumSize;
            return ret;
        }        
    };

    struct ancillary_t 
    {
        double time;
        char scandir;

        template< typename Src > 
        static ancillary_t fromStruct( const Src &x )
        {
            ancillary_t ret;
            ret.time = x.time;
            ret.scandir = x.scandir;
            return ret;
        }        
    };

    struct Ports
    {
        const ancillary_t *anc_in;
        const CacheType *cache_in;
        OutType *hbb_out;
        OutType *abb_out;
        OutType *zbb_out;

        Ports(const ancillary_t *anc_in_, const CacheType *cache_in_, 
                OutType *hbb_out_, OutType *abb_out_, OutType *zbb_out_):
            anc_in(anc_in_),
            cache_in(cache_in_),
            hbb_out(hbb_out_),
            abb_out(abb_out_),
            zbb_out(zbb_out_)
        { }
    };

protected:
    const settings_t settings;
    Ports P;

    class noContextAvailable
    { };

    template< typename Iteratorlike >
    const PCC &sanitized( char scenetype, char scandir, Iteratorlike iter ) // throws noContextAvailable
    {
        const CacheType &cache_in( *P.cache_in );
        // since lower_bound returns the iterator pointing to the first element with a key not less than k,
        // but what we want is the last key which is less than or equal to k!
        if ( (iter != cache_in.begin()) && (iter->first != PCC::key(scenetype,scandir,P.anc_in->time)))
            --iter;

        // make sure that we're returning something that actually is for this scene and direction!
        if (!PCC::sanityCheckCacheEntry( scenetype,scandir,iter )) 
        {
            throw noContextAvailable();
            /// FIXME: WARNING, exception constrained within the capsule or other assertion
            //return nullContext;
        }
        return *(iter->second);
    }

    inline const PCC &invokeBestContextFor( char scenetype ) 
    {
        const CacheType &cache_in( *P.cache_in );
        // find the best context to use by searching the database
        return sanitized( scenetype, P.anc_in->scandir, 
                cache_in.lower_bound( PCC::key(scenetype, P.anc_in->scandir, P.anc_in->time) ) );
    }
    
    // FIXME:
    // Do we interpolate blackbody temperatures as well? YES
    // Do we use the interpolated blackbody temperature as compared to the actual as an error metric? FUTURE
    
    
public:
    ThreeBodyPolynomialCalibrationSynthesis( const settings_t &settings_, const Ports &p_ ): 
        settings(settings_), P(p_) 
    { }

    int operator()( ) 
    {
        assert( settings.spectrumSize <= P.hbb_out->size() );
        assert( settings.spectrumSize <= P.abb_out->size() );
        assert( settings.spectrumSize <= P.zbb_out->size() );
        
        char ctx='-';
        try 
        {
            ctx='H'; invokeBestContextFor(ctx)( *P.hbb_out, P.anc_in->time );
            ctx='A'; invokeBestContextFor(ctx)( *P.abb_out, P.anc_in->time );
            ctx='Z'; invokeBestContextFor(ctx)( *P.zbb_out, P.anc_in->time );
        }
        catch( noContextAvailable nca )
        {
            std::cerr << __FILE__ << ':' << __LINE__ 
                        << ": no context found for " << ctx << " at time "<< P.anc_in->time << std::endl;
            return -1;
        }
        return 0;
    }    
};

} // namespace
