/** 
*  \file PiecewiseLinearCalibrationContext.cc
*  \brief Unit test for Piecewise Linear Calibration Context.
*    
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: PiecewiseLinearCalibrationContextTest.cc,v 1.5.2.4 2005/12/15 23:13:34 rayg Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*  \endverbatim
*/

#include <cassert>
#include <cmath>
#include <cstdlib>
#include <map>
#include <boost/array.hpp>

#include "Physics.h"  //wnums_t
#include "Planck.hxx"
#include "PiecewiseLinearCalibrationContext.hxx" 
#include "BindCdf.hxx"

const unsigned SIZE=2048;

using namespace gips;

typedef PolynomialCalibrationContext::BasicCacheType CacheType;

typedef PlanckRadiances<physics::wnums_t> PlanckSpectrum;

typedef boost::array< std::complex<double>, SIZE > CplxSpectrum;
typedef boost::array< double, SIZE > RealSpectrum;

typedef PiecewiseLinearCalibrationStudy< PlanckSpectrum, CacheType > PLCS_type;
typedef ThreeBodyPolynomialCalibrationSynthesis< CacheType, CplxSpectrum > TBPCS_type;

struct TestBench
{ 
    struct settings_t 
    {
        const unsigned spectrumSize;
        settings_t(): spectrumSize(SIZE) { }
    } settings;
    
    PLCS_type::ancillary_t 
        study_ancil_in;

    // input to study -- planck spectra
    PlanckSpectrum 
        hbb,
        abb,
        zbb,
        obs;
                
    CacheType cache;

    TBPCS_type::ancillary_t 
        synth_ancil_in;
        
    RealSpectrum diskbuffer;
      
    // output of synth -- complex spectra
    CplxSpectrum 
        Shbb,
        Sabb,
        Szbb,
        Sobs;        
};

const char scandir[] = { 'F' };
const unsigned SCENETYPES = 3;
const char scenetype[] = { 'H', 'A', 'Z' };
const double startTemps[] = { 333., 273., 1.4 };
const double startTime[] = { 200., 200., 200. }; // FUTURE: try variable time offsets 210.5, 221.0 }; 

const gips::physics::wnums_t wavenumbers( 1800., 2400., SIZE );

void fillCache( TestBench &tb, FileToMemBinderGroup &shifter, PLCS_type &plcs, double deltaTemp=0.0, double deltaT = 0.0 )
{
    for( unsigned i=0; i<SCENETYPES; i++ )
    {
        tb.obs = PlanckSpectrum( startTemps[i] + deltaTemp, &wavenumbers );
        tb.study_ancil_in.scenetype = scenetype[i];
        tb.study_ancil_in.scandir = scandir[0];
        tb.study_ancil_in.time = startTime[i] + deltaT;        

        // copy the planck spectrum into real memory for disk output -- mildly crufty but necessary for CDF.        
        std::copy( tb.obs.begin(), tb.obs.end(), tb.diskbuffer.begin() );
        
        int rc = plcs();
        assert(rc==0);
        DBG(std::cerr << tb.cache.size() << "elements in cache\n");
        shifter.update(+1,SEEK_CUR);
    }
}

void printCache( TestBench &tb)
{
  CacheType::iterator pos;
  std::cout << "context_cache keys\n";
  for(pos = tb.cache.begin(); 
      pos != tb.cache.end();
      ++pos)
    std::cout << "   (" 
	      << pos->first.first.first 
	      << ", "
	      << pos->first.first.second
	      << "), "
	      << pos->first.second
	      << std::endl;
}

void test1()
{
    TestBench mybench;
    

// study phase        
    {
        int rc = system( "ncgen -b plcc_study.cdl" );
        assert( 0 == rc );

        NcFile fp( "plcc_study.nc", NcFile::Write ); // as defined by plcc_study.cdl
        SimpleCdfRecordMapping mapping;    
        FileToMemBinderGroup shifter;

        long shape[] = { 1, SIZE };
        //shifter.push_back( newCdfVectorWriter( &mybench.cal.spectrum, NcVar_p(fp.get_var("signalReal")), shape, shape+2, &mapping ));
        assert( fp.get_var("signalReal") );
        //assert( fp.get_var("signalImag") );

        shifter.push_back( newCdfVectorWriter( &mybench.diskbuffer, 
                                fp.get_var("signalReal"), shape, shape+2, &mapping ));
        shifter.push_back( newCdfFieldWriter( &mybench.study_ancil_in.time, fp.get_var("time_offset"),  &mapping ));    
        shifter.push_back( newCdfFieldWriter( &mybench.study_ancil_in.scenetype, fp.get_var("scenetype"),  &mapping ));    

        PLCS_type::settings_t study_settings;
        study_settings = PLCS_type::settings_t::fromStruct( mybench.settings );
        PLCS_type mystudy( study_settings, PLCS_type::Ports( &mybench.obs, &mybench.study_ancil_in, &mybench.cache ) );
        
        shifter.update(0,SEEK_SET);
        
        // FIXME: put in a sinusoidal envelope or something like that to perturb the temperature with
        fillCache(mybench,shifter,mystudy);
	fillCache(mybench,shifter,mystudy,1.0, 900 ); // add context information for 900+ seconds later
        fillCache(mybench,shifter,mystudy,4.0, 2700 );

	printCache(mybench);
    }    
    
    { // synth phase 
        int rc = system( "ncgen -b plcc_synth.cdl" );
        assert( 0 == rc );

        NcFile fp( "plcc_synth.nc", NcFile::Write ); 
        SimpleCdfRecordMapping mapping;    
        FileToMemBinderGroup shifter;
        
        long shape[] = { 1, SIZE };
        shifter.push_back( newCdfCplxVectorWriter( &mybench.Shbb, 
                                fp.get_var("hbbReal"), fp.get_var("hbbImag"), shape, shape+2, &mapping ));
        shifter.push_back( newCdfCplxVectorWriter( &mybench.Sabb, 
                                fp.get_var("abbReal"), fp.get_var("abbImag"), shape, shape+2, &mapping ));
        shifter.push_back( newCdfCplxVectorWriter( &mybench.Szbb, 
                                fp.get_var("zbbReal"), fp.get_var("zbbImag"), shape, shape+2, &mapping ));    
        shifter.push_back( newCdfFieldWriter( &mybench.synth_ancil_in.time, fp.get_var("time_offset"),  &mapping ));    

        TBPCS_type::settings_t settings( TBPCS_type::settings_t::fromStruct( mybench.settings ) );

        TBPCS_type::Ports p( &mybench.synth_ancil_in, &mybench.cache, 
                                &mybench.Shbb, &mybench.Sabb, &mybench.Szbb );
        TBPCS_type mysynth(settings, p );
        
        mybench.synth_ancil_in = TBPCS_type::ancillary_t::fromStruct(mybench.study_ancil_in);
        
        mybench.synth_ancil_in.time = startTime[0] - 300.;
        while( mybench.synth_ancil_in.time < (2700. + startTime[0]) )
        {
            // bind output arrays to the record we're writing to 
            shifter.update(+1,SEEK_CUR);
            
            // set up the input 
            mybench.synth_ancil_in.time += 300.;
            
            // turn the crank
            mysynth();
        }
    }
    
}

int main( int argc, char *argv[] )
{
    test1();
}




