/**
* \file Planck.h
 * Planck curve utility library
 *
 * \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>
 *
 * \version $Id: Planck.hxx,v 1.1.2.1 2005/12/15 23:13:34 rayg Exp $
 *
 *  \par Copyright:
 *  \verbatim
 *
 *  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2005
 *  Space Science and Engineering Center
 *  University of Wisconsin - Madison, USA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  \endverbatim
 */

#ifndef H_PLANCK
#define H_PLANCK

#include "Physics.h"

namespace gips
{


class PlanckConstants
{
protected:
    const double h,c,k;
    const double c1, c2;
    
public:
    PlanckConstants():
            h(6.62606876E-34),
            c(2.99792458E8),
            k(1.3806503E-23),
            c1(2.0*h*c*c*1e8),
            c2(h*c/k*1e2)
    { }
    
    PlanckConstants& operator=( const PlanckConstants &x ) { return *this; }
};



template< typename W >
class PlanckRadiances: private PlanckConstants
{
    double T; // Kelvin
    const W *w;

// FIXME: deal with changing constants!
// random access generator??
public:
    PlanckRadiances( double T_ = 0.0, const W *w_ = 0 ): 
            T(T_), 
            w(w_)
    {
    }
        
    typedef double value_type;
    
    inline unsigned size() const { return w==NULL? 0 : w->size(); }
    
    inline bool operator==( const PlanckRadiances<W> &it ) const 
    { return it.T==T && it.w==w; }
    
    inline bool operator!=( const PlanckRadiances<W> &it ) const 
    { return ! (it==*this); }

    inline double operator[]( unsigned offset ) const
    {
        if (!w) return NAN;
        const W &wavenumbers(*w);
        double wn = wavenumbers[offset];
        return 1000.0 * c1 * (wn * wn * wn) / (exp((c2 * wn) / T) - 1.0);
    }

    typedef indexed_array_const_fwd_bwd_iterator< PlanckRadiances<W> > 
        iterator, const_iterator, reverse_iterator, const_reverse_iterator;
    
    inline iterator begin() const { return iterator(*this); }
    inline iterator end() const { return iterator(*this,size()); }
    
    inline iterator rbegin() const { return iterator(*this,0,-1); }
    inline iterator rend() const { return iterator(*this,size(),-1); }

};

template< typename W, typename EmissVector >
class EffectivePlanckRadiances
{
    PlanckRadiances<W> bb, refl;
    const EmissVector &emissivity; // 0.0 - 1.0 emissivity fraction matched to wavenumber scale
public:
    
    // return effective planck radiance for a blackbody of a given temperature Tbb (Kelvin),
    // with a reflected temperature of the optical system observing it being Trefl,
    // and an emissivity of the cavity of emiss. 
    EffectivePlanckRadiances( double Tbb_, double Trefl_, const EmissVector *emiss_, const W *w_ ): 
        bb(Tbb_,w_), refl(Trefl_,w_), emissivity(*emiss_) 
    {
        assert( w_->size() == emiss_->size() ); 
    }
        
    typedef double value_type;
    
    inline unsigned size() const { return bb.size(); }
    
    inline bool operator==( const EffectivePlanckRadiances<W,EmissVector> &it ) const 
    { return it.bb==bb && it.refl==refl && it.emissivity==emissivity; }
    
    inline bool operator!=( const EffectivePlanckRadiances<W,EmissVector> &it ) const 
    { return ! (it==*this); }

    inline double operator[]( unsigned offset ) const
    {
        return bb[offset] * emissivity[offset] + refl[offset] * (1.0 - emissivity[offset]);
    }

    typedef indexed_array_const_fwd_bwd_iterator< EffectivePlanckRadiances<W,EmissVector> > 
        iterator, const_iterator, reverse_iterator, const_reverse_iterator;
    
    inline iterator begin() const { return iterator(*this); }
    inline iterator end() const { return iterator(*this,size()); }
    
    inline iterator rbegin() const { return iterator(*this,0,-1); }
    inline iterator rend() const { return iterator(*this,size(),-1); }


};

} // namespace

#endif
