/** 
*  \file RadiometricCalibration.hxx
*  \brief Template for radiometric calibration equation.
*    
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: RadiometricCalibration.hxx,v 1.6.2.8 2005/12/15 20:34:30 rayg Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*  \endverbatim
*/

 
#ifndef HXX_RADCAL
#define HXX_RADCAL

#include <vector>
#include <complex>
#include <cassert>
#include <cmath>

#include "Planck.hxx"

namespace gips
{

template
<
    typename InType,    // input spectrum buffers, complex
    typename OutType,   // output spectrum buffer, real
    typename WnumsType // vector-compatible type delivering wavenumbers
>
class TwoBodyRadiometricCalibration
{
public:
    struct settings_t
    {
        unsigned spectrumSize;
        
        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.spectrumSize = x.spectrumSize;
            return ret;
        }
    };
    
    struct reference_t
    {
        WnumsType wavenumbers;
    };
    
    struct ancillary_t
    {
        double hbbTemp;
        double abbTemp;

        template< typename Src > 
        static ancillary_t fromStruct( const Src &x )
        {
            ancillary_t ret;
            ret.hbbTemp = x.hbbTemp;
            ret.abbTemp = x.abbTemp;
            return ret;
        }
    };

    struct Ports
    {
        const reference_t *ref_in;
        const ancillary_t *anc_in;
        const InType *obs_in;
        const InType *hbb_in;
        const InType *abb_in;
        OutType *cal_out; // calibrated spectrum (real portion)
        OutType *diag_out; // diagnostic spectrum (imaginary portion)

        Ports(const reference_t *ref_in_, const ancillary_t *anc_in_, 
                const InType *obs_in_, const InType *hbb_in_, const InType *abb_in_, 
                OutType *cal_out_, OutType *diag_out_):
            ref_in(ref_in_),
            anc_in(anc_in_),
            obs_in(obs_in_),
            hbb_in(hbb_in_),
            abb_in(abb_in_),
            cal_out(cal_out_),
            diag_out(diag_out_)
        { }
    };
    
protected:
    const settings_t settings;
    Ports P;
    
public:
    explicit TwoBodyRadiometricCalibration( const settings_t &s_, const Ports &p_ ): settings(s_), P(p_) { }

    void operator()( void )
    {
        const InType& obsIn(*P.obs_in);   // local references to input/output arrays passed in by pointer
        const InType& abbIn(*P.abb_in);
        const InType& hbbIn(*P.hbb_in);
        OutType& calOut(*P.cal_out);
        OutType& diagOut(*P.diag_out);
        
        assert( P.ref_in->wavenumbers.size() == obsIn.size() );
        assert( P.ref_in->wavenumbers.size() == abbIn.size() );
        assert( P.ref_in->wavenumbers.size() == hbbIn.size() );

        PlanckRadiances<WnumsType> Ba( P.anc_in->abbTemp, &P.ref_in->wavenumbers );
        PlanckRadiances<WnumsType> Bh( P.anc_in->hbbTemp, &P.ref_in->wavenumbers );        
        
        for( unsigned w = 0; w < P.ref_in->wavenumbers.size(); w++ )
        {
            std::complex<double> rad = (obsIn[w] - abbIn[w])
                                        / (hbbIn[w] - abbIn[w]);
            calOut[w] = rad.real() * (Bh[w] - Ba[w]) + Ba[w];
            diagOut[w] = rad.imag();
        }
    }
    
}; // template<> class TwoBodyRadiometricCalibration


template
<
    typename InType,    // input spectrum buffers, complex
    typename OutType,   // output spectrum buffer, real
    typename WnumsType, // vector-compatible type delivering wavenumbers
    typename EmissType  // vector-compatible blackbody emissivity 
>
class ThreeBodyRadiometricCalibration
{
public:
    struct settings_t
    {
        unsigned spectrumSize;

        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.spectrumSize = x.spectrumSize;
            return ret;
        }
    };
    
    struct reference_t
    {
        WnumsType wavenumbers;
        EmissType bbEmissivity; // blackbody emissivity characterization
        
        double calmirrorTau;
        double telescopeTau;
        
        template< typename Src > 
        static reference_t fromStruct( const Src &x )
        {
            reference_t ret;
            ret.wavenumbers = x.wavenumbers;
            ret.bbEmissivity = x.bbEmissivity;
            ret.calmirrorTau = x.calmirrorTau;
            ret.telescopeTau = x.telescopeTau;
            return ret;
        }
    };
    
    struct ancillary_t
    {
        double hbbTemp;
        double abbTemp;
        double zbbTemp;
        double reflectedTemp;
        
        template< typename Src > 
        static ancillary_t fromStruct( const Src &x )
        {
            ancillary_t ret;
            ret.hbbTemp = x.hbbTemp;
            ret.abbTemp = x.abbTemp;
            ret.zbbTemp = x.zbbTemp;
            ret.reflectedTemp = x.reflectedTemp;
            return ret;
        }
    };

    struct Ports
    {
        const reference_t *ref_in;
        const ancillary_t *anc_in;
        const InType *obs_in;
        const InType *hbb_in;
        const InType *abb_in;
        const InType *zbb_in;
        OutType *cal_out; // calibrated spectrum (real portion)
        OutType *diag_out; // diagnostic spectrum (imaginary portion)
        
        Ports(const reference_t *ref_in_, const ancillary_t *anc_in_, 
                const InType *obs_in_, const InType *hbb_in_, const InType *abb_in_, const InType *zbb_in_, 
                OutType *cal_out_, OutType *diag_out_):
            ref_in(ref_in_),
            anc_in(anc_in_),
            obs_in(obs_in_),
            hbb_in(hbb_in_),
            abb_in(abb_in_),
            zbb_in(zbb_in_),
            cal_out(cal_out_),
            diag_out(diag_out_)
        { }
    };
    
protected:
    const settings_t settings;
    Ports P;

public:

    explicit ThreeBodyRadiometricCalibration( const settings_t &s_, const Ports &p_ ): settings(s_), P(p_) { }

    void operator()( void )
    {
        assert( settings.spectrumSize == P.ref_in->wavenumbers.size() );
        assert( P.obs_in->size() >= settings.spectrumSize );
        assert( P.abb_in->size() >= settings.spectrumSize );
        assert( P.hbb_in->size() >= settings.spectrumSize );
        assert( P.zbb_in->size() >= settings.spectrumSize );        
        
        // NOTE: the use of interpolated BB temp versus actual BB temp 
        // We want to start by using the actual measurement, though this may need to be filtered due to 
        // possible noise in the BB temp measurement.. Also, may want to use interpolated vs real BB temp 
        // as a quality metric or correction term to the calibration model.
        
        // We salt the planck radiances to get effective planck radiances
        // This B'(wavenumber) = bbemissivity[wavenumber] * B(wavenumber,bbtemp) + (1-bbemissivity[wavenumber]) * B(wavenumber,reflectedtemp)
        // bbemissivity[wavenumber] is ground-characterized reference data computed using monte-carlo model
        // emissivity of space is expected to be 1.0
        
        // NOTE: The blackbody temperatures input into this system are synthesized in the same manner as the 
        // complex BB spectra; therefore the actual blackbody temperature is not relevant to the calibration.
        // The actual blackbody temperature and its deviation from the interpolated/extrapolated/synthesized form
        // may be used as an error term, correction or quality metric of the calibration study model. 
        
        EffectivePlanckRadiances<WnumsType, EmissType> 
            Bc( P.anc_in->abbTemp, P.anc_in->reflectedTemp, &P.ref_in->bbEmissivity, &P.ref_in->wavenumbers );
        EffectivePlanckRadiances<WnumsType, EmissType> 
            Bh( P.anc_in->hbbTemp, P.anc_in->reflectedTemp, &P.ref_in->bbEmissivity, &P.ref_in->wavenumbers );        
        PlanckRadiances<WnumsType> 
            Bs( P.anc_in->zbbTemp, &P.ref_in->wavenumbers );        

        double netTau = (P.ref_in->calmirrorTau / P.ref_in->telescopeTau);
        
        for( unsigned w = 0; w < P.ref_in->wavenumbers.size(); w++ )
        {
            /* N = (Tm/Tt) * (Bh-Bc) * Re( (Ce-Cs) / (Ch-Cc) ) + Bs */            
            
            std::complex<double> CCCC = (P.obs_in->operator[](w) - P.zbb_in->operator[](w))
                                        / (P.hbb_in->operator[](w) - P.abb_in->operator[](w));
            double TTBH = netTau * (Bh[w] - Bc[w]);
            
            std::complex<double> rad = CCCC * TTBH + Bs[w];
            
            P.cal_out->operator[](w) = rad.real();
            P.diag_out->operator[](w) = rad.imag();
        }
    }    
}; // template<> class ThreeBodyRadiometricCalibration



}; // namespace gips

#endif
 