/** 
*  \file RadiometricCalibrationTest.cc
*  \brief Unit test for radiometric calibration equation.
*    
*
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: RadiometricCalibrationTest.cc,v 1.4.2.4 2005/12/15 20:34:30 rayg Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*  \endverbatim
*/

#include <cassert>
#include <boost/array.hpp>

#include "Physics.h"  //wnums_t
//#include "Planck.h"
#include "RadiometricCalibration.hxx" 

using namespace std;
using namespace gips;

const unsigned SIZE=2048;

typedef boost::array< complex< double >, SIZE > CplxSpectrum;
typedef boost::array< double, SIZE > RealSpectrum;

// simplified two-body calibration model without blackbody emissivity spectrum
typedef TwoBodyRadiometricCalibration< CplxSpectrum, RealSpectrum, physics::wnums_t > CalTwoBody;

// three-body calibration includes as reference an emissivity spectrum
typedef ThreeBodyRadiometricCalibration< CplxSpectrum, RealSpectrum, physics::wnums_t, vector< double > > CalThreeBody;


struct TestBench2
{
    // input
    CalTwoBody::reference_t ref;
    CalTwoBody::ancillary_t ancil;    
    CplxSpectrum hbb, abb, obs;
    
    // output
    RealSpectrum cal, diag;
        
    CalTwoBody::settings_t _calSettings()
    {
        CalTwoBody::settings_t ret;
        ret.spectrumSize = SIZE;
        return ret;
    }
        
    CalTwoBody::Ports _calPorts()
    {
        return CalTwoBody::Ports( &ref, &ancil, &obs, &hbb, &abb, &cal, &diag );
    }        
};

struct TestBench3
{
    // input
    CalThreeBody::reference_t ref;
    CalThreeBody::ancillary_t ancil;    
    CplxSpectrum hbb, abb, zbb, obs;
    
    // output
    RealSpectrum cal, diag;

    CalThreeBody::settings_t _calSettings()
    {
        CalThreeBody::settings_t ret;
        ret.spectrumSize = SIZE;
        return ret;
    }
        
    CalThreeBody::Ports _calPorts()
    {
        return CalThreeBody::Ports( &ref, &ancil, &obs, &hbb, &abb, &zbb, &cal, &diag );
    }    
};


template< typename SpecType >
void copyPlanck( SpecType &S, double kelvin, const physics::wnums_t &wn )
{
    gips::PlanckRadiances<physics::wnums_t> B(kelvin,&wn);
    std::copy( B.begin(), B.end(), S.begin() );
}


// initialize a test bench with a numerically reasonable but extremely simple test case
void setupSimple( TestBench2 &tb )
{
    tb.ref.wavenumbers = physics::wnums_t( 1800., 2400., tb.obs.size() );
    //tb.ref.bbemissivity = vector<double>( tb.obs.cplxspectrum.size(), 1.0 );

    // create synthetic spectra to be calibrated against
    copyPlanck( tb.abb, 273.15, tb.ref.wavenumbers );
    copyPlanck( tb.hbb, 333.0, tb.ref.wavenumbers );
    //copyPlanck( tb.zbb, 1.7 /*FIXME*/, tb.ref.wavenumbers );

    copyPlanck( tb.obs, 300.0, tb.ref.wavenumbers );
    //tb.ref.telescopeTau = 1.0;
    //tb.ref.calmirrorTau = 1.0;
}

void setupSimple( TestBench3 &tb )
{
    tb.ref.wavenumbers = physics::wnums_t( 1800., 2400., tb.obs.size() );
    tb.ref.bbEmissivity = vector<double>( tb.obs.size(), 1.0 );

    // create synthetic spectra to be calibrated against
    copyPlanck( tb.abb, 273.15, tb.ref.wavenumbers );
    copyPlanck( tb.hbb, 333.0, tb.ref.wavenumbers );
    copyPlanck( tb.zbb, 1.7 /*FIXME*/, tb.ref.wavenumbers );

    copyPlanck( tb.obs, 300.0, tb.ref.wavenumbers );
    tb.ref.telescopeTau = 1.0;
    tb.ref.calmirrorTau = 1.0;
}



void test2()
{
    TestBench2 mybench;
    
    setupSimple(mybench);
    
    // hook it up!
    CalTwoBody CalStage( mybench._calSettings(), mybench._calPorts() );
    
    // generate output!
    CalStage();
}


void test3()
{
    TestBench3 mybench;
    
    setupSimple(mybench);
    
    // hook it up!
    CalThreeBody CalStage( mybench._calSettings(), mybench._calPorts() );
    
    // generate output!
    CalStage();
}


int main( int argc, char *argv[] )
{
    test2();
    test3();
}
