/** 
*  \file SimCal2Test.cc
*  \brief Unit test for calibration for simulation data
*    
*  \author G. Martin <graemem@ssec.wisc.edu>
*
*  \version $Id: SimCal2Test.cc,v 1.1.2.19 2005/12/16 21:45:00 graemem Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* FUTURE: consider parallelizing file I/O. Possibly group readers and 
* writers separately within a Binder Group and specify no sharing of 
* resources among different binders to prevent race conditions. Binders
* can themselves be multi-threaded if desired.
*
*  \endverbatim
*/



#include <cassert>
#include <list>
#include <algorithm>
#include <string>
#include <vector>
#include <iostream> // std::cerr
#include <fstream>
#include <math.h>

// NETCDF includes
#include "netcdf.hh"
#include "netcdfcpp.h"

#include "Physics.h"  //wnums_t
#include "Planck.hxx"

#include "IfgToSpectrum.hxx"
//#include "PiecewiseLinearCalibrationContext.hxx" 
#include "RadiometricCalibration.hxx"
#include "BindCdf.hxx"  // bring in source data and reference settings

using namespace std;
using namespace gips;

const double HBB_TEMP = 300.0;
const double ABB_TEMP = 265.0;
const double ZBB_TEMP = 2.76;

const unsigned IFG_SIZE=2048;   // FIXME: read from file
const unsigned SPC_SIZE=IFG_SIZE/2;
const unsigned NUM_RECS = 16384;
typedef vector< complex <double> > CplxSpectrum;
typedef vector<double> RealSpectrum;

typedef IfgToSpectrum< RealSpectrum, CplxSpectrum > Ifg2Spc;

// three-body calibration includes as reference an emissivity spectrum
typedef ThreeBodyRadiometricCalibration< CplxSpectrum, RealSpectrum, physics::wnums_t, vector< double > > CalThreeBody;

typedef NcVar *NcVar_p; // NOT! boost::shared_ptr<NcVar> NcVar_p; since NcFile retains ownership!

// Data is loaded into the test bench, which is connected to the assortment of components needed
// to achieve the processing goal. Each component has limited visibility of the work area, with
// sub-structures satisfying their connection requirements. This reduces copying and dynamic allocation
// substantially, and permits "syntax-compatible" data containers to be used according to the situation;
// for instance, concurrency-controlled containers could be considered in the case that the
// components (e.g. the input service) are operating in multiple threads.

struct TestBench
{   
    // Ifg2Spectrum variables: 4 capsules
    RealSpectrum obs;               // connectors
    CplxSpectrum obsSpectrum;   
    Ifg2Spc::settings_t obsI2sSettings;   // settings
    Ifg2Spc::Ports obsI2sPorts;           // ports

    RealSpectrum abb;
    CplxSpectrum abbSpectrum;
    Ifg2Spc::settings_t abbI2sSettings;   // settings
    Ifg2Spc::Ports abbI2sPorts;           // ports
   
    RealSpectrum zbb;
    CplxSpectrum zbbSpectrum;
    Ifg2Spc::settings_t zbbI2sSettings;   // settings
    Ifg2Spc::Ports zbbI2sPorts;           // ports
   
    RealSpectrum hbb;
    CplxSpectrum hbbSpectrum;
    Ifg2Spc::settings_t hbbI2sSettings;   // settings
    Ifg2Spc::Ports hbbI2sPorts;           // ports
   
   // Three body cal variables
    CalThreeBody::reference_t ref;
    CalThreeBody::ancillary_t ancil;    
    RealSpectrum cal, diag; 
    CalThreeBody::settings_t calSettings;  //settings      
    CalThreeBody::Ports calPorts;      // ports


    static physics::wnums_t wavenumbersLW( ) // from ROK, wavenumbers for spectral form; move to CDF reference file
    {
        const unsigned IfgSize = IFG_SIZE;
        const double vlaser = 9398.0; // effective laser wavenumber
        const double r_lw = 8; // reduction factor
        const double n_lw = IfgSize;
        const unsigned SpcSize = IfgSize/2;
        
        double x_lw = (n_lw/2.0) * r_lw/vlaser;
        double dv = 1.0 / (2.0 * x_lw);
        double v0_lw = (n_lw/2 + 1) * dv;
        physics::wnums_t ret; // FIXME: make a friendlier constructor
        ret.baseWnum = v0_lw;
        ret.deltaWnum = dv;
        ret.wnumCount = SpcSize;
        return ret;
    }

    // ctor
    TestBench():obs(IFG_SIZE), obsSpectrum(IFG_SIZE), obsI2sPorts(&obs, &obsSpectrum ),
                abb(IFG_SIZE), abbSpectrum(IFG_SIZE), abbI2sPorts(&abb, &abbSpectrum ),
                zbb(IFG_SIZE), zbbSpectrum(IFG_SIZE), zbbI2sPorts(&zbb, &zbbSpectrum ),
                hbb(IFG_SIZE), hbbSpectrum(IFG_SIZE), hbbI2sPorts(&hbb, &hbbSpectrum ),
                cal(SPC_SIZE), diag(SPC_SIZE), 
                calPorts( &ref, &ancil, &obsSpectrum, &hbbSpectrum, &abbSpectrum, &zbbSpectrum, &cal, &diag )
        {
            obsI2sSettings.interferogramSize = IFG_SIZE;   
            obsI2sSettings.chopRawSpectrumFrom = IFG_SIZE/2;
            obsI2sSettings.chopRawSpectrumTo = IFG_SIZE;
            obsI2sSettings.wrapRawSpectrumLastPoint = false;
            // obsI2sSettings.spectrumSize = SPC_SIZE;

            abbI2sSettings.interferogramSize = IFG_SIZE;   
            abbI2sSettings.chopRawSpectrumFrom = IFG_SIZE/2;
            abbI2sSettings.chopRawSpectrumTo = IFG_SIZE;
            abbI2sSettings.wrapRawSpectrumLastPoint = false;
            // abbI2sSettings.spectrumSize = SPC_SIZE;

            zbbI2sSettings.interferogramSize = IFG_SIZE;   
            zbbI2sSettings.chopRawSpectrumFrom = IFG_SIZE/2;
            zbbI2sSettings.chopRawSpectrumTo = IFG_SIZE;
            zbbI2sSettings.wrapRawSpectrumLastPoint = false;
            // zbbI2sSettings.spectrumSize = SPC_SIZE;

            hbbI2sSettings.interferogramSize = IFG_SIZE;   
            hbbI2sSettings.chopRawSpectrumFrom = IFG_SIZE/2;
            hbbI2sSettings.chopRawSpectrumTo = IFG_SIZE;
            hbbI2sSettings.wrapRawSpectrumLastPoint = false;
            // hbbI2sSettings.spectrumSize = SPC_SIZE;

            // FIXME : why don't spectrum sizes match?
            calSettings.spectrumSize = SPC_SIZE;
            ref.wavenumbers = wavenumbersLW( );
//            cout << "wavenumbers: base" << ref.wavenumbers.baseWnum 
//                    << " interval " << ref.wavenumbers.deltaWnum << endl;
            ref.bbEmissivity.clear();
            ref.bbEmissivity.resize( SPC_SIZE, 1.0 );
            ref.telescopeTau = 1.0;
            ref.calmirrorTau = 1.0;
            ancil.abbTemp = ABB_TEMP;
            ancil.hbbTemp = HBB_TEMP;
            ancil.zbbTemp = ZBB_TEMP;
            ancil.reflectedTemp = 200.0;
        }    
};

// Not fully tested.
// extract the part of the string after the last '/'. If there
// there is no '/', return the entire string. If string ends
// in '/' return string with length 0;
string extractFilenameUnix(const string& str)
{
    unsigned slashPos = str.rfind('/');
    if(slashPos == (str.length()))
        return string(str); 
    else if(slashPos == (str.length()-1))
        return string(""); 
    else 
        return string(str.substr(slashPos, str.length()-1));
}

int test( const std::vector<std::string>& Hfiles, const std::vector<std::string>& Afiles, const std::vector<std::string>& Zfiles, const std::vector<std::string>& Efiles)
{
    TestBench myBench;

    // create output file, as awkwardly as possible
    string calFileName("test/");
    calFileName.append(extractFilenameUnix(Efiles[0]).c_str());
    calFileName.append(".cal.nc");
    string uncalFileName("test/");
    uncalFileName.append(extractFilenameUnix(Efiles[0]).c_str());
    uncalFileName.append(".uncal.nc");

    string ncgenCommand( "ncgen -b test/template.cdl -o ");
    int rc = system( (ncgenCommand + calFileName).c_str() );
    assert( 0 == rc );
    rc = system( (ncgenCommand + uncalFileName).c_str() );
    assert( 0 == rc );

    // bind connectors to data files. FUTURE: accomplish this with a connector class
    NcFile fpObs( Efiles[0].c_str(), NcFile::ReadOnly ); 
    NcFile fpAbb( Afiles[0].c_str(), NcFile::ReadOnly ); 
    NcFile fpZbb( Zfiles[0].c_str(), NcFile::ReadOnly ); 
    NcFile fpHbb( Hfiles[0].c_str(), NcFile::ReadOnly ); 

    NcFile fpCal( calFileName.c_str(), NcFile::Write ); 
    NcFile fpUncal( uncalFileName.c_str(), NcFile::Write ); 

    string tagName("$Id: SimCal2Test.cc,v 1.1.2.19 2005/12/16 21:45:00 graemem Exp $");
    fpCal.add_att("generated_by", tagName.c_str());
    fpUncal.add_att("generated_by", tagName.c_str());

    GiftsPixelRecordMapping mapping;    
    FileToMemBinderGroup shifter;
    long shape[] = { 1, 1, IFG_SIZE };
    shifter.push_back( newCdfVectorReader( &myBench.obs, NcVar_p(fpObs.get_var("signalReal_lw")), shape, shape+3, &mapping ));
    shifter.push_back( newCdfVectorReader( &myBench.abb, NcVar_p(fpAbb.get_var("signalReal_lw")), shape, shape+3, &mapping ));
    shifter.push_back( newCdfVectorReader( &myBench.zbb, NcVar_p(fpZbb.get_var("signalReal_lw")), shape, shape+3, &mapping ));
    shifter.push_back( newCdfVectorReader( &myBench.hbb, NcVar_p(fpHbb.get_var("signalReal_lw")), shape, shape+3, &mapping ));
    
    shifter.push_back( newCdfCplxVectorWriter( &myBench.obsSpectrum, NcVar_p(fpUncal.get_var("signalReal_lw")), NcVar_p(fpUncal.get_var("signalImag_lw")), shape, shape+3, &mapping ));
    shifter.push_back( newCdfVectorWriter( &myBench.cal, NcVar_p(fpCal.get_var("signalReal_lw")), shape, shape+3, &mapping ));
    shifter.push_back( newCdfVectorWriter( &myBench.diag, NcVar_p(fpCal.get_var("signalImag_lw")), shape, shape+3, &mapping ));
    
    Ifg2Spc i2sObs(myBench.obsI2sSettings, myBench.obsI2sPorts);
    Ifg2Spc i2sAbb(myBench.abbI2sSettings, myBench.abbI2sPorts);
    Ifg2Spc i2sZbb(myBench.hbbI2sSettings, myBench.hbbI2sPorts);
    Ifg2Spc i2sHbb(myBench.zbbI2sSettings, myBench.zbbI2sPorts);

    CalThreeBody c3b(myBench.calSettings, myBench.calPorts);

    for(unsigned i=0; i<NUM_RECS; i++)
    {
        shifter.update( +1, SEEK_CUR );
                      
        i2sObs();
        i2sAbb();
        i2sZbb();
        i2sHbb();
         
        c3b();
    }
    return 0;
}

void printUsageMsg()
{
    std::cout 
        << "Usage: \n" 
        << "   SimCal2Test.unittest [filetype] [filename] [filetype] [filename] ...\n" 
        << "where [filetype] is one of the values:\n"
        << "   H     - hot blackbody data cube\n"
        << "   W     - warm blackbody data cube\n"
        << "   S     - space blackbody data cube\n"
        << "   E     - Earth observation data cube\n"
        << "Options list must specify one of each blackbody and one observation cube.\n";
}

int main( int argc, char *argv[] )
{
    //physics::wnums_t wnu( TestBench::wavenumbersLW() );
    //cout << "wavenumbers: base " << wnu.baseWnum 
    //        << " interval " << wnu.deltaWnum 
    //        << " count " << wnu.wnumCount << endl;

    // options list must be pairs of options and values
    if(argc%2 != 1) 
    {
        printUsageMsg();
        return 1;
    }
    
    std::vector<std::string> Hfiles, Afiles, Zfiles, Efiles;
    
    for(int i=1; i<argc; i+=2)
    {
        if(std::string("H") == (argv[i])) Hfiles.push_back(std::string(argv[i+1]));
        else if(std::string("W") == (argv[i])) Afiles.push_back(std::string(argv[i+1]));
        else if(std::string("S") == (argv[i])) Zfiles.push_back(std::string(argv[i+1]));
        else if(std::string("E") == (argv[i])) Efiles.push_back(std::string(argv[i+1]));
        else 
        {
            printUsageMsg();
            return 1;
        }
    }
    
    // must have exactly one of each blackbody file and one obs file
    if((Hfiles.size() != 1) || 
       (Afiles.size() != 1) || 
       (Zfiles.size() != 1) || 
       (Efiles.size() != 1))
    {
        printUsageMsg();
        return 1;
    }

    return test(Hfiles, Afiles, Zfiles, Efiles);
}
