/** 
*  \file Physics.h
*  \brief contains definitions in the Physics namespace
*  \author R.K.Garcia <rayg@ssec.wisc.edu>
*
*  \version $Id: Physics.h,v 1.15 2005/11/05 22:38:24 rayg Exp $
*
*  \par Copyright:
*  \verbatim
*
*  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
*  Space Science and Engineering Center
*  University of Wisconsin - Madison, USA
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*  \endverbatim
*/


#ifndef H_PHYSICS
#define H_PHYSICS

#include <cmath>
#include <iterator>

/**
*  \brief Physics namespace, including standard types and formulae for 
*  radiances, wavenumbers, temperatures, times, voltages.
*/
namespace gips 
{
// FIXME: move this to a better place
    template< typename IA >
    class indexed_array_const_fwd_bwd_iterator
    {
      const IA &source;
      unsigned i;
      const int incr;
      
      public:
      indexed_array_const_fwd_bwd_iterator( const IA &source_, unsigned i_=0, int incr_=1 ): 
              source(source_), i(i_), incr(incr_)
      { }
      
      typename IA::value_type operator*() const { return source[i]; }
      indexed_array_const_fwd_bwd_iterator& operator++() { i+=incr; return *this; }
      indexed_array_const_fwd_bwd_iterator& operator++(int) { i+=incr; return *this; }
      indexed_array_const_fwd_bwd_iterator& operator--() { i-=incr; return *this; }
      indexed_array_const_fwd_bwd_iterator& operator--(int) { i-=incr; return *this; }
      
      inline bool operator==( const indexed_array_const_fwd_bwd_iterator &it ) const
      {
          return it.source==source && i==it.i;
      }
      
      inline bool operator!=( const indexed_array_const_fwd_bwd_iterator &it ) const
      {
          return it.source!=source || i!=it.i;
      }
      
      inline int operator-( const indexed_array_const_fwd_bwd_iterator &it ) const
      {
          return i - it.i;
      }
      
      typedef std::bidirectional_iterator_tag iterator_category;
      typedef typename IA::value_type value_type;
      typedef typename IA::value_type *pointer;
      typedef typename IA::value_type &reference;
      typedef int difference_type;
    };

    namespace physics 
    { 
	
	/**
	 * floating point time representation
	 * Units: seconds
	 */
	typedef double seconds_t;
	
	/**
	 * floating point temperature representation
	 * Units: Kelvin
	 */
	typedef double kelvin_t;
	
	/**
	 * floating point wavenumber representation
	 * Units: wavenumbers
	 */
	typedef double wnum_t;
	
	/** 
	 *inverse of wavenumbers - centimeters.
	 */
	typedef double cm_t;
	
	/** 
	 * Radiance_t value
	 * Units: 
	 */
	typedef double radiance_t;
	
	
	/** \fun planckRadiance
	 * \param T temperature in Kelvin
	 * \param wnum wavenumber
	 * \todo Implement me!
	 */
//	inline radiance_t planckRadiance( kelvin_t T, wnum_t wnum )
//	{
//	    return radiance_t( 0. ); 
//	};

	
	/** standardized view of regularly-spaced wavenumbers */
	struct wnums_t
	{
		typedef double value_type;
	    value_type baseWnum;
	    value_type deltaWnum;
	    unsigned wnumCount;

        wnums_t( double start_=0.0, double end_=1.0, unsigned count_=0 ): 
            baseWnum(start_), deltaWnum((end_ - start_)/double(count_-1)), wnumCount(count_) { }
            
	    /** truncate to bounds. FIXME: this should raise an exception if at all possible */
	    inline unsigned trunc( unsigned offset ) const 
		{ return (offset>=wnumCount)? wnumCount-1 : offset; }
	    
	    /** access as if it were an array */
	    inline double operator[]( unsigned offset ) const
		{ return deltaWnum*double(trunc(offset))+baseWnum ; }
		
		inline unsigned size() const { return wnumCount; }
		
		inline bool operator==( const wnums_t &it ) const 
		{ return it.baseWnum==baseWnum && it.deltaWnum==deltaWnum && it.wnumCount==wnumCount; }
		
		inline bool operator!=( const wnums_t &it ) const 
		{ return ! (it==*this); }
		
		typedef indexed_array_const_fwd_bwd_iterator<wnums_t> 
            iterator, const_iterator, reverse_iterator, const_reverse_iterator;
		
		inline iterator begin() const { return iterator(*this); }
		inline iterator end() const { return iterator(*this,wnumCount); }
		
		inline iterator rbegin() const { return iterator(*this,0,-1); }
		inline iterator rend() const { return iterator(*this,wnumCount,-1); }
	};
	
// FIXME - send this to the great library in the sky...		
	/** \brief Transforms a Wnums_t class into a blackbody radiance spectrum
	 * Future: this can create a cache full of NANs internally, and return values from the cache
	 * instead of recomputing the power every time an access occurs. 
	 * Possibly as an alternate implementation of the same interface, e.g. CachingPlanckRadianceGenerator
	 */
//	class PlanckRadianceGenerator
//	{
//	    const wnums_t &W;
//	    const kelvin_t T;
//	    
//	public:
//	    PlanckRadianceGenerator( const wnums_t& _w, const kelvin_t& _t ): 
//		W(_w), T(_t)
//	    { };
//		
//		inline radiance_t operator[]( unsigned offset ) const
//		{
//		    return planckRadiance( T, W[offset] ); 
//		};
//	};
//	
    } // namespace physics
} // namespace gips


#endif

