/**
 * \file CplxFFT.h
 * \brief performs complex FFT
 * \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>
 *
 * based directly on the Scanning-HIS rsh2fbf application, written by Ray Garcia <rayg@ssec.wisc.edu>
 *
 *  \version $Id: CplxFFT.hxx,v 1.9.2.4 2005/12/14 22:34:08 rayg Exp $
 *
 *  \par Copyright:
 *  \verbatim
 *
 *  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
 *  Space Science and Engineering Center
 *  University of Wisconsin - Madison, USA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  \endverbatim
 */
 

#ifndef H_CPLXFFT
#define H_CPLXFFT

#ifndef NO_DEPENDENCY_INCLUDES

// the FFTW system-dependent #include polka
#ifdef __APPLE__
#include "dfftw_threads.h"
#include "drfftw_threads.h"
#elif defined(__CYGWIN32__)
#include "fftw.h"
#include "rfftw.h"
#define fftw_threads_init() 
#define fftw_threads_one(a,b,c,d) fftw_one(b,c,d)
#define rfftw_threads_one(a,b,c,d) rfftw_one(b,c,d)
#else
#include "fftw_threads.h"
#include "rfftw_threads.h"
#endif

// GIPS includes

#include <complex>
#include <vector>

#endif


namespace gips {

template
<
    typename InType,
    typename OutType
>
class CplxFFT 
{
public:
    struct settings_t 
    {
        unsigned interferogramSize;  // number of values in the interferogram

        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.interferogramSize = x.interferogramSize;
            return ret;
        }        
    };
    
    struct Ports 
    {
        const InType *obs_in;
        OutType *obs_out;
        
        explicit Ports(         
            const InType *obs_in_,
            OutType *obs_out_
            ):
               obs_in(obs_in_),
               obs_out(obs_out_)
        {
        }
    };

protected:
    const settings_t settings;
    Ports P;

    fftw_plan plan; // cached machine-dependent plan - should it be const?
    
    // fftw is potentially destructive on input buffer
    std::vector< std::complex<double> > tempin, tempout;
    
    // fft to an intermediate buffer and transform to the final output
    template< typename OutIter >
    void anyfft( OutIter out )
    {
        tempout.resize(settings.interferogramSize);
        fftw_one( plan, (fftw_complex *)&tempin[0], (fftw_complex *)&tempout[0] );
        std::copy( tempout.begin(), tempout.end(), out );
    }
    
    // template function specialization for the case where no transformation is needed
    template< std::complex< double > * >
    void anyfft( std::complex< double > *out )
    {
        fftw_one( plan, &tempin[0], out );
    }

public:
    explicit CplxFFT(const settings_t &s, const Ports &p): 
        settings(s), P(p), tempin(s.interferogramSize)
    {
        plan = fftw_create_plan( settings.interferogramSize, FFTW_FORWARD, FFTW_ESTIMATE ); 
        // alternatively, use FFTW_ESTIMATE instead of FFTW_MEASURE for faster startup
    
        static bool has_initted_threads = false;
        if (!has_initted_threads)
        {
            fftw_threads_init();    
            has_initted_threads = true;
        }        
    }; // ctor

    ~CplxFFT() 
    {
        fftw_destroy_plan( plan );
    }; // dtor

    int operator()( void )
    {    
        assert( P.obs_out->size() >= settings.interferogramSize );
        assert( P.obs_in->size() >= settings.interferogramSize );
    
        std::copy( P.obs_in->begin(), P.obs_in->begin() + settings.interferogramSize, tempin.begin() );
        OutType &obs_out( *P.obs_out );
        anyfft( &obs_out[0] );
        return 0;
    }

}; // class CplxFFT

typedef CplxFFT< std::vector< std::complex< double > >, std::vector< std::complex< double > > > BasicCplxFFT;
    
} // namespace
#endif
    
    

