/**
 * \file IfgShift.h
 * \brief Shifts raw interferogram to form expected by the FFT operator.
 * \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>
 *
 * based directly on the Scanning-HIS rsh2fbf application, written by Ray Garcia <rayg@ssec.wisc.edu>
 *
 *  \version $Id: IfgShift.hxx,v 1.7.2.2 2005/12/14 20:41:52 rayg Exp $
 *
 *  \par Copyright:
 *  \verbatim
 *
 *  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
 *  Space Science and Engineering Center
 *  University of Wisconsin - Madison, USA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  \endverbatim
 */



#ifndef H_IFGSHIFT
#define H_IFGSHIFT

#ifndef NO_DEPENDENCY_INCLUDES
// GIPS includes
#include <complex>
#include <vector>

#endif

namespace gips {

template< typename InType, typename OutType >
class IfgShift
{    
public:
    struct settings_t
    {
        unsigned interferogramSize;
                
        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.interferogramSize = x.interferogramSize;
            return ret;
        }
    };
    
    struct Ports
    {
        const InType *obs_in;
        OutType *obs_out;
        
        Ports( const InType *in_, OutType *out_ ): obs_in(in_), obs_out(out_) { }
    };

protected:
    const settings_t settings;
    // instance variable containing a copy of the connectors
    Ports P;    

    template< typename A, typename B >
    void anyShift( B &out, const A &in )
    {
        unsigned N(settings.interferogramSize), halfN(N/2); 
        unsigned src(0), dest(0);
        
        for( src = halfN; src<N; src++,dest++ ) 
            out[dest] = in[src];
    
        for( src=0; src<halfN; src++, dest++ ) 
            out[dest] = in[src]; 
    }
        
public:
    
    explicit IfgShift(const settings_t &s, const Ports &p): settings(s), P(p)
    {       
    }
    
    ~IfgShift() 
    { 
    }
    
    // The meat of the template. 
    int operator() ( )
    {
        assert( P.obs_in->size() >= settings.interferogramSize );
        assert( P.obs_out->size() >= settings.interferogramSize );
    
        //const settings_t& settings(ports.settings.getForInput());
        anyShift( *P.obs_out, *P.obs_in );
        
        return 0;    
    } 
}; // class 

typedef IfgShift< std::vector< std::complex<double> >, std::vector< std::complex<double> > > BasicIfgShift;


} // namespace





#endif
    
