/*
 * \file Cfg2SpectrumTest.cc
 * \brief unit test for SHISStage subclass of Ifg2SpectrumStage
 * \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>
 *
 *  \version $Id: IfgToSpectrumTest.cc,v 1.1.2.3 2005/12/14 23:35:51 rayg Exp $
 *
 *  \par Copyright:
 *  \verbatim
 *
 *  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
 *  Space Science and Engineering Center
 *  University of Wisconsin - Madison, USA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  \endverbatim
 */

// C++ STL includes
#include <string>
#include <vector>
#include <iostream> // std::cerr
#include <fstream>
#include <math.h>

// GIPS framework includes
#include "IfgToSpectrum.hxx"

// NETCDF includes
#include "netcdf.hh"
#include "netcdfcpp.h"

using namespace std;
using namespace gips;


const unsigned SIZE=2048;
typedef vector< complex< double > > data_vec; // could also be boost::array< complex<double>, SIZE >
typedef IfgToSpectrum< data_vec, data_vec > Ifg2Spc;



typedef NcVar *NcVar_p; // NOT! boost::shared_ptr<NcVar> NcVar_p; since NcFile retains ownership!

// fill buf with a sin wave f(i) = scaleFactor*sin(((2*pi*i)/period) + phaseShift)
// period is number of data points in each sin wave
void genSinWav(std::vector<float>& buf, 
	       const float offset, 
	       const float scaleFactor, 
	       const int period,
	       const double phaseShift = 0.0) 
{
    
  for(unsigned i=0; i<buf.size(); i++) buf[i] = offset + scaleFactor*(std::sin((2*M_PI*i/period) + phaseShift));

};

// v1 gets sum vector
bool addVectors(std::vector<float>& v1, const std::vector<float>& v2)
{
  if(v1.size() != v2.size()) return false;
  for(unsigned i=0; i<v1.size(); i++) v1[i] += v2[i];
  return true;
}

// fill real component of complex array from float array, zero fill imaginary 
// component. 
bool composeComplex(const std::vector<float>& srcReal, const std::vector<float>& srcImag, data_vec& dest) 
{
  if(srcReal.size() != dest.size()) return false; 
  if(srcImag.size() != dest.size()) return false; 
  for(unsigned i=0; i<srcReal.size(); i++) dest[i] = data_vec::value_type(srcReal[i], srcImag[i]);
  return true;
}

void decomposeComplex(const data_vec& src, 
		    std::vector<float>& reals, 
		    std::vector<float>& imags ) 
{
  for(unsigned i=0; i<src.size(); i++)
    {
      reals[i] = src[i].real();
      imags[i] = src[i].imag();
    }
} 

// will stomp destination file if fexists
bool copyFile(const std::string& srcFilename, 
              const std::string& dstFilename) 
{ 
    std::ifstream src((srcFilename).c_str()); 
    std::ofstream dst((dstFilename).c_str()); 

    if( !src || !dst ) return false;

    std::copy(std::istreambuf_iterator<char>(src), 
              std::istreambuf_iterator<char>(), 
              std::ostreambuf_iterator<char>(dst)); 

    src.close();
    dst.close();
    return true;
} 

bool writeIfg( const std::vector<float>& ifg,
	       const std::string& filename, 
	       int recNum)
{    
    NcFile ncfile(filename.c_str(), NcFile::Write );

    if(!ncfile.is_valid()) return false;

    NcVar_p var(ncfile.get_var("signalReal_lw"));

    var->set_cur( recNum%4, recNum/4, 0 );
    var->put( &ifg[0], 1, 1, ifg.size() );
    
    return true;
}

bool writeIfg( const data_vec &ifg,
	       const std::string& filename, 
	       int recNumReal,
	       int recNumImag)
{    
  std::vector<float> floatVector(ifg.size());
  for(unsigned i=0; i<ifg.size(); i++) floatVector[i] = ifg[i].real();
  if(writeIfg(floatVector, filename, recNumReal) != true) return false;
  for(unsigned i=0; i<ifg.size(); i++) floatVector[i] = ifg[i].imag();
  return writeIfg(floatVector, filename, recNumImag);  
}

bool readIfg( std::vector<float>& ifg,
	       const std::string& filename, int recNum)
{
    NcFile ncfile( filename.c_str(), NcFile::ReadOnly );
    
    if(!ncfile.is_valid()) return false;

    NcVar_p var(ncfile.get_var("signalReal_lw"));

    var->set_cur( recNum%4, recNum/4, 0 );
    var->get( &ifg[0], 1, 1, ifg.size() );
     
    return true;
}

// fill real component of complex array from float array, zero fill imaginary 
// component. 
bool composeComplex(const std::vector<float>& src, data_vec &dest) 
{
  if(src.size() != dest.size()) return false; 
  for(unsigned i=0; i<src.size(); i++) dest[i] = data_vec::value_type(src[i], 0);
  return true;
}


Ifg2Spc::settings_t test_settings()
{
  Ifg2Spc::settings_t s_;
  s_.interferogramSize = SIZE;
  s_.chopRawSpectrumFrom = SIZE/2;
  s_.chopRawSpectrumTo = SIZE;
  s_.wrapRawSpectrumLastPoint = false;
  return s_;
}    

int testIfg(const data_vec& inputIfg, data_vec& outputIfg)
{
  Ifg2Spc::settings_t s_(test_settings());
   
  if(inputIfg.size() != s_.interferogramSize) return -1;
  if(outputIfg.size() != s_.interferogramSize) return -1;
  

  // convert input array of type complex<float> to type complex<double>. No
  // loss of precision so conversion is done implicitly by complex class.
  //for( unsigned i=0; i<inputIfg.size(); i++) i_.interferogram[i] = inputIfg[i]; 

  Ifg2Spc::Ports ports( &inputIfg, &outputIfg );

  Ifg2Spc op(s_,ports);
  int res = op();

  // convert output array of type complex<double> to type complex<float>. 
  // Possible loss of precision, so conversion must be done explicitly.
  //for( unsigned i=0; i<outputIfg.size(); i++ ) outputIfg[i] = std::complex <float>(o_.cplxspectrum[i]);
  
  return res;
}

int sinTest()
{   
  // create an interferogram with a sin pattern. for now use double
  // type instead of complex to match the current file format.
  std::vector<float> buf1(SIZE);
  std::vector<float> buf2(SIZE); 
  data_vec inputIfg(SIZE);
  data_vec outputIfg(SIZE);

  const std::string templateFile = "test/template.cdf";
  const std::string outputFile = "helix.cdf";  
  //create a test file from a template
  copyFile(templateFile, outputFile);
  
  // ****** TEST 1: sin wave: 2 waves with different periods ******
  genSinWav(buf1, 0.0, 1.0e+06, 128);
  genSinWav(buf2, 0.0, 1.0e+06, 128, M_PI/2); 
  //  addVectors(buf1, buf2);

  composeComplex(buf1, buf2, inputIfg);
  
  // write raw sin data to record 0
  // if(!writeIfg(inputIfg, outputFile,0)) return Operator::ERROR;
  if(!writeIfg(inputIfg, outputFile, 0, 1)) return -1;

  int res = testIfg(inputIfg, outputIfg);
  if(res != 0) return res;  

  if(!writeIfg(outputIfg, outputFile, 2, 3)) return -1;

  return 0;
};

int livelinessTest()
{
    data_vec i(SIZE,1.), o(SIZE,0.);

    // invoke operator with the correct ports
    Ifg2Spc::Ports ports(&i,&o);

    Ifg2Spc op(test_settings(),ports);
    int res = op();
    
    return res;
}; // testOperator()


int main()
{
    int rc;
    //rc = livelinessTest();
    rc = sinTest();
    assert(rc == 0);
    return(0);
}

