/**
 * \file SpectrumChop.h
 * \author Maciek Smuga-Otto <maciek@ssec.wisc.edu>
 *
 *  \version $Id: SpectrumChop.hxx,v 1.6.2.4 2005/12/14 23:35:51 rayg Exp $
 *
 *  \par Copyright:
 *  \verbatim
 *
 *  Copyright UW/SSEC, ALL RIGHTS RESERVED, 2004
 *  Space Science and Engineering Center
 *  University of Wisconsin - Madison, USA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  \endverbatim
 */

#ifndef H_SPECCHOP
#define H_SPECCHOP

#ifndef NO_DEPENDENCY_INCLUDES

#include <vector>
#include <string>
#include <complex>

#endif

// NOTE: This file is an example of the kind of output expected from the 
// GIPS flanging compiler, and will in the future NOT be generated by hand.
// As a result, all the input/output structs are Plain Old Data (POD)s,
// as indicated by the _t suffixes. Furthermore, all their members have
// names such as would be published in the global data dictionary.


namespace gips 
{

template
<
    typename InType,
    typename OutType
>   
class SpectrumChop
{
public:
    struct settings_t 
    {
        unsigned int chopRawSpectrumFrom; 
        unsigned int chopRawSpectrumTo;
        bool wrapRawSpectrumLastPoint;

        template< typename Src > 
        static settings_t fromStruct( const Src &x )
        {
            settings_t ret;
            ret.chopRawSpectrumFrom = x.chopRawSpectrumFrom;
            ret.chopRawSpectrumTo = x.chopRawSpectrumTo;
            ret.wrapRawSpectrumLastPoint = x.wrapRawSpectrumLastPoint;
            return ret;
        }
    };
    
    struct Ports 
    {
        const InType *obs_in;
        OutType *obs_out;
        
        explicit Ports(         
            const InType *obs_in_,
            OutType *obs_out_
            ):
               obs_in(obs_in_),
               obs_out(obs_out_)
        {
        }
    };

protected:
    const settings_t settings;
    Ports P;
                 
public:    

    explicit SpectrumChop(const settings_t &s, const Ports &p): settings(s), P(p)
    {    
        // check consistency of symbols defined in settings. Namely:
        // chopping obs_in in range [chopRawSpectrumFrom,chopRawSpectrumTo)

        assert(settings.chopRawSpectrumTo > settings.chopRawSpectrumFrom);
        assert((settings.chopRawSpectrumTo - settings.chopRawSpectrumFrom) <= P.obs_out->size());
        assert(P.obs_in->size() >= settings.chopRawSpectrumTo);
    }
    
    int operator()(void)
    {    
        unsigned dest(0);
        for (unsigned int src(settings.chopRawSpectrumFrom); 
                src < settings.chopRawSpectrumTo; 
                ++src, ++dest)
        {
            P.obs_out->operator[](dest) = P.obs_in->operator[](src);
        };
        
        // curious case of the last point: wrap it, or zero?
        P.obs_out->operator[](dest - 1) = 
            settings.wrapRawSpectrumLastPoint ? P.obs_in->operator[](settings.chopRawSpectrumFrom) :
                                                0.0;

        //cerr << "SpectrumChop: dataOut@600: " << obs_out.rawSpectrum[600] << endl; // DEBUG     
        return 0;
    }

}; // class SpectrumChop
        

typedef SpectrumChop
        < 
            std::vector< std::complex<double> >,  // input buffer type 
            std::vector< std::complex<double> >   // output buffer type
        >
        BasicSpectrumChop;
            
}; // namespace


#endif
